/*********************************************************************************************************//**
 * @file    BMK52T016_HT32/src/BMK52T016.c
 * @version V1.0.1
 * @date    2024-08-06
 * @brief   The function of BM62S2201_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMK52T016.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMK52T016_WIRE = BMK52T016_WIRE;
/* Private variables ---------------------------------------------------------------------------------------*/
/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief module wire number select.
 * @param wire_number: select wire number.
 * @retval BMK52T016_SUCCESS/BMK52T016_FAILURE
 ************************************************************************************************************/
BMK52T016_selStatus BMK52T016_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BMK52T016_FAILURE;
  }

  gBMK52T016_WIRE = wire_number;
  return BMK52T016_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BMK52T016_Init(void)
{
	I2CMaster_Init(gBMK52T016_WIRE, BMK52T016_MASTERADDR, BMK52T016_CLKFREQ);
	
	//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BMK52T016_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BMK52T016_INTPIN_GPIO_ID, BMK52T016_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BMK52T016_INTPIN_GPIO_PORT, BMK52T016_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BMK52T016_INTPIN_GPIO_PORT, BMK52T016_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BMK52T016_INTPIN_GPIO_PORT, BMK52T016_INTPIN_GPIO_PIN, ENABLE);
}

/*********************************************************************************************************//**
 * @brief  get Key Status
 * @param  void
 * @retval Returns the INT state  
            @arg  0:INT output low level  press
            @arg  1:INT output high level   unpress 
 ************************************************************************************************************/
uint8_t BMK52T016_getINT(void)
{
	return (GPIO_ReadInBit(BMK52T016_INTPIN_GPIO_PORT, BMK52T016_INTPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  read Key Value
 * @param  void
 * @retval keyValue:Variables for storing KeyValue data
                  @arg 0:No key is pressed
                  @arg bit0=1 : key1 is pressed
                  @arg bit1=1 : key2 is pressed
                  @arg ...
                  @arg bit15=1 : key16 is pressed
 ************************************************************************************************************/
uint16_t BMK52T016_readKeyValue(void)
{
	  uint16_t keyValue = 0;
    uint8_t BMK52T016_KeyCMD[1]={0X0D};
    uint8_t BMK52T016_tempL[1],BMK52T016_tempH[1];
    BMK52T016_writeBytes(BMK52T016_KeyCMD,1); 
    _BMK52T016_delay(5);
    BMK52T016_readBytes(BMK52T016_tempL,BMK52T016_REQUESTLEN1);
    _BMK52T016_delay(5);
    BMK52T016_readBytes(BMK52T016_tempH,BMK52T016_REQUESTLEN1);
    keyValue = (BMK52T016_tempH[0]<<8)+BMK52T016_tempL[0];
    _BMK52T016_delay(10);
    return keyValue;
}

/*********************************************************************************************************//**
 * @brief  get 16  key trigger threshold
 * @param  buff :Store the obtained 16 keystroke trigger threshold
                  @arg buff[0]:Store the acquired KEY1 trigger threshold
                  @arg buff[1]:Store the acquired KEY2 trigger threshold
                  @arg ...........................................
                  @arg buff[15]:Store the acquired KEY16 trigger threshold
 * @retval void
 ************************************************************************************************************/
void BMK52T016_getThreshold(uint8_t buff[])
{
	  uint8_t BMK52T016_TLCMD[1] = {0xD8};//threshold CMD
    BMK52T016_writeBytes(BMK52T016_TLCMD,1); 
    _BMK52T016_delay(5); 
    BMK52T016_readBytes(buff,16); 
    _BMK52T016_delay(10);
}

/*********************************************************************************************************//**
 * @brief  Set the trigger threshold of 16 keys
 * @param  buff :The triggering thresholds of 16 keys need to be set
                  @arg buff[0]:Store the acquired KEY1 trigger threshold,range: 10-64
                  @arg buff[1]:Store the acquired KEY2 trigger threshold,range: 10-64
                  @arg ...........................................
                  @arg buff[15]:Store the acquired KEY16 trigger threshold,range: 10-64  
 * @retval Implementation status:
              @arg 0:BMK52T016_R_SUCCESS 
              @arg 1:BMK52T016_R_FAIL
 ************************************************************************************************************/
int BMK52T016_setThreshold(uint8_t buff[])
{
    uint16_t BMK52T016_total = 0;
    uint8_t BMK52T016_TXbuff[18] = {0};
		uint8_t i=0;
    for(i=0;i<16;i++)                    //sum
    {
        BMK52T016_total = BMK52T016_total + buff[i];
        BMK52T016_TXbuff[i+1] = buff[i];
    }
    if(160 <= BMK52T016_total && BMK52T016_total<= 1024)//10~64
    {
        BMK52T016_TXbuff[0] = 0XD8;
        BMK52T016_TXbuff[17] = (uint8_t)BMK52T016_total;
        BMK52T016_writeBytes(BMK52T016_TXbuff,18);
        _BMK52T016_delay(10);
        return BMK52T016_R_SUCCESS;
    }
    else
    {
      return BMK52T016_R_FAIL;
    }
}

/*********************************************************************************************************//**
 * @brief  set Threshold Level of 16 keys
 * @param  Level :Storing ThresholdLevel Data,range: 0-6   
 * @retval Implementation status:
              @arg 0:BMK52T016_R_SUCCESS 
              @arg 1:BMK52T016_R_FAIL 
 ************************************************************************************************************/
int BMK52T016_setAllThresholdLevel(uint8_t level)
{
		uint8_t BMK52T016_thr = 0;
		uint8_t BMK52T016_sum =  0;
	  uint8_t BMK52T016_STLCMD[20] = {0};
	  uint8_t i = 0;
    if(level>6)
    {
        return BMK52T016_R_FAIL;
    }
    BMK52T016_thr = 10+level*9;
    BMK52T016_sum = 16*BMK52T016_thr;
		BMK52T016_STLCMD[0] = 0xD8;
		for(i = 1;i<17;i++)
		{
			BMK52T016_STLCMD[i] = BMK52T016_thr;
		}
		BMK52T016_STLCMD[17] = BMK52T016_sum;
    BMK52T016_writeBytes(BMK52T016_STLCMD,BMK52T016_REQUESTLEN18);
    _BMK52T016_delay(10);
    return BMK52T016_R_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  writeBytes
 * @param  wbuf :the bytes sent
             wlen :the length of the data sent  
 * @retval void
 ************************************************************************************************************/
void BMK52T016_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	  uint8_t sendbuf[255] = {0};
		uint8_t i = 0;
		for(i=0;i<wlen;i++)
		{
			sendbuf[i] = wbuf[i];
		}
		
	{
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = BMK52T016_MASTERADDR;
    gI2CMaster.Tx.puBuffer = (uint8_t *)&sendbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;

    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBMK52T016_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBMK52T016_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
	}
}

/*********************************************************************************************************//**
 * @brief  read Bytes
 * @param  rbuf :the bytes receive
             rlen :the length of the data receive
 * @retval void
 ************************************************************************************************************/
void BMK52T016_readBytes(uint8_t rbuf[], uint8_t rlen)
{
     I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
     I2CMaster_Typedef I2CMaster = { 0 };

     I2CMaster.uSlaveAddress = BMK52T016_MASTERADDR;
     I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
     I2CMaster.Rx.uLength = rlen;
     I2CMaster.uTimeout_us = 30000;
     I2CMaster.isDynamicOnOff_I2C = FALSE;

     I2CMaster_Read(gBMK52T016_WIRE, &I2CMaster);
	
     do {
       I2CResult = I2CMaster_GetStatus(gBMK52T016_WIRE);
     } while (I2CResult == I2C_MASTER_GOING);
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BMK52T016_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}


/* Private functions ---------------------------------------------------------------------------------------*/

