/*********************************************************************************************************//**
 * @file    BM22S4221_1_HT32/src/BM22S4221_1.c
 * @version V1.0.1
 * @date    2024-09-23
 * @brief   The function of BM22S4221_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM22S4221_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM22S4221_1_SERIAL = BM22S4221_1_SERIAL;
uint8_t BM22S4221_1_dataBuff[25];
/* Private variables ---------------------------------------------------------------------------------------*/
uint8_t _BM22S4221_1_len;     //Length of data reveviced
uint8_t _BM22S4221_1_state;   
uint8_t _BM22S4221_1_sum;
uint8_t _BM22S4221_1_checksum;//Checksum of data reveviced     
uint8_t _BM22S4221_1_res;
/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief module serial number select.
 * @param serial_number: select serial number.
 * @retval BM22S4221_1_FAILURE/BM22S4221_1_SUCCESS
 ************************************************************************************************************/
BM22S4221_1_selStatus BM22S4221_1_selSerial(uint32_t serial_number)
{
  if(CHECK_SERIALn(serial_number) == -1)
  {
    return BM22S4221_1_FAILURE;
  }

  gBM22S4221_1_SERIAL = serial_number;
  return BM22S4221_1_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using UART communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM22S4221_1_Init(void)
{

	USART_InitTypeDef USART_InitStructure = {0};
  USART_InitStructure.USART_BaudRate = 9600;
  USART_InitStructure.USART_WordLength = USART_WORDLENGTH_8B;
  USART_InitStructure.USART_StopBits = USART_STOPBITS_1;
  USART_InitStructure.USART_Parity = USART_PARITY_NO;
  USART_InitStructure.USART_Mode = USART_MODE_NORMAL;
  UARTM_Init(gBM22S4221_1_SERIAL, &USART_InitStructure, BM22S4221_1_UART_TXTIMEOUT);
	
	//INPUT:STA Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BM22S4221_1_STAPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BM22S4221_1_STAPIN_GPIO_ID, BM22S4221_1_STAPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BM22S4221_1_STAPIN_GPIO_PORT, BM22S4221_1_STAPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BM22S4221_1_STAPIN_GPIO_PORT, BM22S4221_1_STAPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BM22S4221_1_STAPIN_GPIO_PORT, BM22S4221_1_STAPIN_GPIO_PIN, ENABLE);
	
}

/*********************************************************************************************************//**
 * @brief Get STATUS pin level
 * @param void
 * @retval 1: Alarm status
           0: Normal status
 ************************************************************************************************************/
uint8_t BM22S4221_1_getSTATUS(void)
{
	return (GPIO_ReadInBit(BM22S4221_1_STAPIN_GPIO_PORT, BM22S4221_1_STAPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief Proactively sending instructions to retrieve data packets
 * @param buff(25byte): Store the FW version and production date
 * @retval 1: module data acquisition failed, there is no correct feedback value
           0: Module data obtained successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_requestInfoPackage(uint8_t buff[])
{
	uint8_t uniCmd[4] = {0xAc, 0x00, 0x00, 0x54};
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,25) == 0x00 && BM22S4221_1_dataBuff[4] == 0xAc)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Query the FW version
 * @param  void
 * @retval FWVer:16bit FWVer
 ************************************************************************************************************/
uint16_t BM22S4221_1_getFWVer(void)
{
  uint16_t FWVer=0;
  uint8_t uniCmd[4] = {0xAD, 0x00, 0x00, 0x53};
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd,4);
	_BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,12) == 0x00 && BM22S4221_1_dataBuff[4] == 0xAD)
  {
    FWVer=(BM22S4221_1_dataBuff[6]<<8 | BM22S4221_1_dataBuff[7]);
  }
  return FWVer;
}
/*********************************************************************************************************//**
 * @brief Query the production date.
 * @param buff[0]:year  buff[1]:month  buff[2]:day
 * @retval 1: module data acquisition failed, there is no correct feedback value
           0: Module data obtained successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_getProDate(uint8_t buff[])
{
  uint8_t uniCmd[4] = {0xAD, 0x00, 0x00, 0x53};
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd,4);
	_BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,12) == 0x00 && BM22S4221_1_dataBuff[4] == 0xAD)
  {
    buff[0]=BM22S4221_1_dataBuff[8];
    buff[1]=BM22S4221_1_dataBuff[9];
    buff[2]=BM22S4221_1_dataBuff[10];
    return   0;
  }
  else
  {
    return   1;
  }
}
/*********************************************************************************************************//**
 * @brief Query whether the serial port data output of the current device is enabled
 * @param void
 * @retval state:1/0
           @arg  0: Serial port TX disable
           @arg  1: Serial port TX enable
					 @arg 2:	error
 ************************************************************************************************************/
uint8_t BM22S4221_1_isAutoTx(void)
{
  uint8_t uniCmd[4] = {0xd0, 0x1b,0x00,0x15};
  uint8_t state=2;
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd,4);
	_BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[6] == BM22S4221_1_AUTO)
  {
    state = 1;
  }
  if (BM22S4221_1_dataBuff[6] == BM22S4221_1_PASSIVE)
  {
    state = 0;
  }
  return state;
}
/*********************************************************************************************************//**
 * @brief Query the normal output of equipment alarm output level
 * @param void
 * @retval ActiveMode:
           @arg  0: Status output low level, normal state is high level
           @arg  1: Status output high level, normal low level
				   @arg	 2:	error
 ************************************************************************************************************/
uint8_t BM22S4221_1_getStatusPinActiveMode(void)
{  
  uint8_t uniCmd[4] = {0xd0, 0x1c,0x00,0x14};
  uint8_t ActiveMode=2;
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd,4);
	_BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[6] == BM22S4221_1_HIGH_LEVEL)
  {
    ActiveMode = 1;
  }
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[6] == BM22S4221_1_LOW_LEVEL)
  {
    ActiveMode = 0;
  }
  return  ActiveMode;
}
/*********************************************************************************************************//**
 * @brief Query internal VBG voltage a/d value
          1.25V/VDD*256=VBG a/d value
 * @param void
 * @retval 0: module data read error
           data:Vbgad internal Ad value of BG voltage
 ************************************************************************************************************/
uint8_t BM22S4221_1_getVBG(void)
{
  uint8_t uniCmd[4] = {0xd2, 0x4c,0x00,0xe2};
  uint8_t VBG=0;
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd,4);
	_BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xd2)
  {
  VBG = BM22S4221_1_dataBuff[6];
  }
  return  VBG;
}
/*********************************************************************************************************//**
 * @brief Determine whether the data packet is automatically output
 * @param void
 * @retval true:is 25 byte data
           false:Incorrect data
 ************************************************************************************************************/
bool BM22S4221_1_isInfoAvailable(void)
{
  uint8_t header[5] = {0xAA,0x19, 0x31, 0x01, 0xAC};
  uint8_t recBuf[25] = {0};
  uint8_t i, num = 0, readCnt = 0, failCnt = 0, checkCode = 0;
  bool isHeader = false, result = false;
	_BM22S4221_1_clearBuf();
  num = UARTM_GetReadBufferLength(gBM22S4221_1_SERIAL);
  /* Serial buffer contains at least one 32-byte data */
  if (num >= 25)
  {
    while (failCnt < 2) // Didn't read the required data twice, exiting the loop
    {
      /* Find 5-byte data header */
			BM22S4221_1_readBytes(recBuf,5);
      for (i = 0; i < 5;)
      {
        if (recBuf[i] == header[i])
        {
          isHeader = true; // Fixed code is correct
          i++;             // Next byte
        }
        else if (recBuf[i] != header[i] && i > 0)
        {
          isHeader = false; // Next fixed code error
          failCnt++;
          break;
        }
        else if (recBuf[i] != header[i] && i == 0)
        {
          readCnt++; // 0xAA not found, continue
        }
        if (readCnt >= (num - 5))
        {
          readCnt = 0;
          isHeader = false; //
          break;
        }
      }
      /* Find the correct data header */
      if (isHeader)
      {
        for (i = 0; i < 5; i++)
        {
          checkCode += recBuf[i]; // Sum checkCode
        }
				BM22S4221_1_readBytes(recBuf,20);
        checkCode = checkCode - recBuf[24];
        checkCode = (~checkCode) + 1; // Calculate checkCode
        /* Compare whether the check code is correct */
        if (checkCode == recBuf[24])
        {
          for (i = 0; i < 25; i++) // True, assign data to _recBuf[]
          {
            BM22S4221_1_dataBuff[i] =recBuf[i];
          }
          result = true;
          break; // Exit "while (failCnt < 2)" loop
        }
        else
        {
          failCnt++; // Error, failCnt plus 1, return "while (failCnt < 2)" loop
          checkCode = 0;
        }
      }
		}
  }
  return result;
}
/*********************************************************************************************************//**
 * @brief Read the judged data packet
 * @param array[]:25 byte data
 * @retval void
 ************************************************************************************************************/
void BM22S4221_1_readInfoPackage(uint8_t array[])
{
	uint8_t i = 0;
	for (i = 0; i < 25; i++)
  {
    array[i] = BM22S4221_1_dataBuff[i];
  }
}
/*********************************************************************************************************//**
 * @brief Send instructions to reset the chip in the module
 * @param void
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_resetModule(void)
{
  uint8_t uniCmd[4] = {0xa0, 0x00, 0x00, 0x60};
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xa0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Send command to restore the module to factory settings
               Factory settings:
               1. internal OPA gain control: maximum gain 37
               2. alarm deviation: 15
               3. alarm detection delay: 3S
               4. alarm signal output: 3S
               5. preheating time: 30s
               6. serial port automatic output de energization
               7. alarm signal output level is high
 * @param void
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_restoreDefault(void)
{
  uint8_t uniCmd[4] = {0xaf, 0x00, 0x00, 0x51};
	_BM22S4221_1_clearBuf();
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(50);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xaf)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify the device serial port data output Disable
 * @param state:PASSIVE 0x00 / AUTO 0x08
       @arg BM22S4221_1_AUTO:Automatic output of enabling module TX pin
       @arg BM22S4221_1_PASSIVE:Automatic output de energization of enabling module TX pin
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setAutoTx(uint8_t state)
{
  uint8_t uniCmd[4] = {0xe0,0x1b,0x00,0x51};
	uint8_t a=0;
	_BM22S4221_1_clearBuf();
	uniCmd[2]=state;
  a=0xe0 + 0x1b + state;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify device alarm output level
 * @param state:BM22S4221_1_HIGH_LEVEL/BM22S4221_1_LOW_LEVEL
        @arg BM22S4221_1_HIGH_LEVEL:When alerting, Status outputs HIGH level, and the normal state is LOW level
        @arg BM22S4221_1_LOW_LEVEL:When alerting, Status outputs LOW level, and the normal state is HIGH leve
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setStatusPinActiveMode(uint8_t state)
{
  uint8_t uniCmd[4] = {0xe0, 0x1c, 0x00, 0x51};
  uint8_t a=0;
  _BM22S4221_1_clearBuf();
	uniCmd[2]=state;
  a=0xe0 + 0x1c + state;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify Internal OPA Gain
 * @param value:setting range is 0~31
             OPA gain=128 + value*8
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setOpaGain(uint8_t value)
{
  uint8_t uniCmd[4] = {0xe0, 0x05, 0x00, 0x51};
  uint8_t a=0;
	_BM22S4221_1_clearBuf();
	uniCmd[2]= value;
  a=0xe0 + 0x05 + value;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify detection deviation value
					deviation value:Change amount of alarm detection
 * @param Threshold:Setting range 15~120
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setAlarmThreshold(uint8_t Threshold)
{
  uint8_t uniCmd[4] = {0xe0, 0x07, 0x00, 0x51};
  uint8_t a=0;
	_BM22S4221_1_clearBuf();
	uniCmd[2]= Threshold;
  a=0xe0 + 0x07 + Threshold;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify alarm detection delay time  
 * @param time:Alarm detection delay time = n*0.5s=time
					unit s.
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setAlarmDetectDelay(uint8_t time)
{
  uint8_t uniCmd[4] = {0xe0, 0x08, 0x00, 0x51};
  uint8_t a=0;
	_BM22S4221_1_clearBuf();
  time=time * 2;
	uniCmd[2]= time;
  a=0xe0 + 0x08 + time;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify the output time of alarm signal status pin   
 * @param time:Alarm status pin output time = n*0.5s=time  unit:s.
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setAlarmOutputTime(uint8_t time)
{
  uint8_t uniCmd[4] = {0xe0, 0x09, 0x00, 0x51};
	uint8_t a=0;
	_BM22S4221_1_clearBuf();
  time=time * 2;
	uniCmd[2]= time;
  a=0xe0 + 0x09 + time;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief Modify preheating time
             Change to less than 30s  
 * @param time:The setting range is 30~127 and the preheating time is not repairable
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 ************************************************************************************************************/
uint8_t BM22S4221_1_setPreheaTime(uint8_t time)
{
  uint8_t uniCmd[4] = {0xe0, 0x0c, 0x00, 0x51};
	uint8_t a=0;
	_BM22S4221_1_clearBuf();
  time=time * 2;
	uniCmd[2]= time;
  a=0xe0 + 0x0c + time;
  uniCmd[3] = ~(a) + 1;
  BM22S4221_1_writeBytes(uniCmd, 4);
  _BM22S4221_1_delay(100);
  if (BM22S4221_1_readBytes(BM22S4221_1_dataBuff,8) == 0x00 && BM22S4221_1_dataBuff[4] == 0xe0)
  {
    return  0;
  }
  else
  {
    return  1;
  }
}
/*********************************************************************************************************//**
 * @brief  write data through UART.
 * @param  wbuf:Variables for storing Data to be sent
 * @param  wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void BM22S4221_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	  UARTM_Write(gBM22S4221_1_SERIAL,wbuf,wlen);
}

/*********************************************************************************************************//**
 * @brief  read data through UART.
 * @param  rbuf:Variables for storing Data to be obtained
 * @param  rlen:Length of data to be obtained
 * @retval BM22S4221_1_READ_OK/BM22S4221_1_READ_ERROR/BM22S4221_1_NO_ACK/BM22S4221_1_TIMEOUT_ERROR 
 ************************************************************************************************************/
uint8_t BM22S4221_1_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	uint8_t j = 0;
	uint8_t _lenCount = 0;
	uint8_t _delayCount = 0;
	_BM22S4221_1_state=BM22S4221_1_NO_ACK;
	for(_lenCount = 0; _lenCount < rlen; _lenCount++)
	{
		_delayCount = 0;
		while(UARTM_GetReadBufferLength(gBM22S4221_1_SERIAL) == 0)
		{
			if(_delayCount >BM22S4221_1_UART_RXTIMEOUT_BYTE)
			{
				return BM22S4221_1_TIMEOUT_ERROR;
			}
			_BM22S4221_1_delay(1);
			_delayCount++;
		}
		UARTM_ReadByte(gBM22S4221_1_SERIAL,rbuf+_lenCount);
	}
	if(BM22S4221_1_dataBuff[0]==0xAA)//checkSum
	{
		_BM22S4221_1_checksum = BM22S4221_1_dataBuff[rlen-1];
		for(j = 0; j < rlen-1; j++)
    {
       _BM22S4221_1_sum += BM22S4221_1_dataBuff[j];
    }
		_BM22S4221_1_sum = ~_BM22S4221_1_sum + 1;
		if(_BM22S4221_1_sum == _BM22S4221_1_checksum)   //checksum correct!
    {
      _BM22S4221_1_sum = 0;
      _BM22S4221_1_state = BM22S4221_1_READ_OK;     //return READ_OK
    }
		else                     //checksum correct!
    {
      _BM22S4221_1_sum = 0;
      _BM22S4221_1_state = BM22S4221_1_READ_ERROR;   //return READ_ERROR
    }
	}
	return _BM22S4221_1_state;
}

/* Private functions ---------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief  clear Buff.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void _BM22S4221_1_clearBuf(void)
{
	int a = 0;
  for(a = 0; a < 12; a++)
  {
    BM22S4221_1_dataBuff[a] = 0;
  } 
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BM22S4221_1_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}




