/*********************************************************************************************************//**
 * @file    BMD58T280_HT32/src/TFT_driver_ILI9341.c
 * @version V1.0.1
 * @date    2025-07-18
 * @brief   The function of ILI9341 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "TFT_driver_ILI9341.h"
#include "ht32_board_config.h"
#include "EBI.h"
#include "font2.h"
#include "delay.h"
#include "SPI.h"
#include "stdlib.h"

u16 Color_Text = 0xFFFF;  // default: White
u16 Color_Back = 0x0000;  // default: Black
u8 textsize = 3;          // default: 3
//default
u16 _TFT_width = 240;
u16 _TFT_height = 320;
u16 MAC = 0x48;
u8 TFT_DIRECTION=0;
#define pgm_read_byte(addr) (*(const unsigned char *)(addr))

/*********************************************************************************************************//**
 * @brief Write Reg 8bit.
 * @param Value: Reg Value.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteReg8(u8 Value)
{
  #if (LCD_MODE==LCD_EBI_MODE)
      EBI_TFT->EBI_TFT_REG = Value;
  #elif (LCD_MODE==LCD_SPI_MODE)
      LCD_CS_LOW; 
      LCD_DC_LOW;
      SPI1_MASTER_SendByte(Value);
      LCD_CS_HIGH;
  #endif
    
}

/*********************************************************************************************************//**
 * @brief Write RAM 8bit.
 * @param Value: RAM Value.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteRAM8(u8 Value)
{
  #if (LCD_MODE==LCD_EBI_MODE)
      EBI_TFT->EBI_TFT_RAM = Value;
  #elif (LCD_MODE==LCD_SPI_MODE)
      LCD_CS_LOW; 
      LCD_DC_HIGH;
      SPI1_MASTER_SendByte(Value);
      LCD_CS_HIGH;
  #endif  
}

/*********************************************************************************************************//**
 * @brief Write RAM 16bit.
 * @param Value: RAM Value.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteRAM16(u16 Value)
{
  #if (LCD_MODE==LCD_EBI_MODE)
      EBI_TFT->EBI_TFT_RAM = Value>>8;
      EBI_TFT->EBI_TFT_RAM = Value;
  #elif (LCD_MODE==LCD_SPI_MODE)
      LCD_CS_LOW; 
      LCD_DC_HIGH;
      SPI1_MASTER_SendByte(Value>>8);
      SPI1_MASTER_SendByte(Value);
      LCD_CS_HIGH;
  #endif  
}

/*********************************************************************************************************//**
 * @brief Write Reg.
 * @param TFT_Reg_Index: Adrr.
          TFT_Reg_Value: Reg Value.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteReg(u32 TFT_Reg_Index, u32 TFT_Reg_Value)
{
  EBI_TFT->EBI_TFT_REG = TFT_Reg_Index;
  EBI_TFT->EBI_TFT_RAM = TFT_Reg_Value;
}

/*********************************************************************************************************//**
 * @brief Write RAM Prior.
 * @param void.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteRAMPrior(void)
{
  TFT_WriteReg8(0x2C);
}

/*********************************************************************************************************//**
 * @brief Write RAM.
 * @param RGB_Set:RGB Value.
 * @retval void
 ************************************************************************************************************/
void TFT_WriteRAM(u32 RGB_Set)
{
  TFT_WriteRAM16(RGB_Set & 0xFFFF);
}

/*********************************************************************************************************//**
 * @brief Set address window
 * @param Xstart : The horizontal position
          Ystart : The vertical position
          Width : The width of the window
          Height : The height of the window
 * @retval void
 ************************************************************************************************************/
void TFT_setAddrWindow(u16 Xstart, u16 Ystart, u16 Width, u16 Height)
{

  TFT_WriteReg8(0x2A);
  TFT_WriteRAM16(Xstart);
  TFT_WriteRAM16(Xstart + Width-1);
  
  TFT_WriteReg8(0x2B);
  TFT_WriteRAM16(Ystart);
  TFT_WriteRAM16(Ystart + Height-1);    
  
  TFT_WriteRAMPrior();

}

/*********************************************************************************************************//**
 * @brief draw one Point
 * @param Xpos    : The horizontal position
          Ypos    : The vertical position
          RGB_Set : Color
 * @retval void
 ************************************************************************************************************/
void TFT_drawPoint(u16 Xpos, u16 Ypos, u32 RGB_Set)
{
  TFT_setAddrWindow(Xpos, Ypos, 1, 1);
  TFT_WriteRAM(RGB_Set);
}

/*********************************************************************************************************//**
 * @brief draw one Point
 * @param Xpos    : The horizontal position
          Ypos    : The vertical position
          RGB_Set : Color
* @retval : void
 * @other Point color: set by the stroke function
 ************************************************************************************************************/
void TFT_point(u16 Xpos, u16 Ypos)
{
	TFT_setAddrWindow(Xpos, Ypos, 1, 1);
	TFT_WriteRAM(Color_Text);
}

/*********************************************************************************************************//**
 * @brief Fill a rectangle
 * @param Xstart : The horizontal position where the line starts
          Ystart : The vertical position where the line starts
          Width  : The width of the rectangle
          Height : The height of the rectangle
          Color  : pattern color
 * @retval void
 ************************************************************************************************************/
void TFT_fillRect(u16 Xstart, u16 Ystart, u16 Width, u16 Height, u32 Color)
{
  u32 i = 0, j = 0;
  TFT_setAddrWindow(Xstart,Ystart,Width,Height);
  for (i = 0; i < Height; i++)
  {
    for (j = 0; j < Width; j++)
    {
      TFT_WriteRAM(Color);
    }
  }
}

/*********************************************************************************************************//**
 * @brief set background
 * @param Color : background color
 * @retval void
 ************************************************************************************************************/
void TFT_background(u16 Color)
{
  TFT_fillRect(0, 0, _TFT_width, _TFT_height, Color);
}

/*********************************************************************************************************//**
 * @brief set The Interior Color
 * @param Color : Interior Color
 * @retval void
 ************************************************************************************************************/
void TFT_fill(u16 Color)
{
  Color_Back = Color;
}

/*********************************************************************************************************//**
 * @brief Set the color of the border/font
 * @param Color : border/font Color
 * @retval void
 ************************************************************************************************************/
void TFT_stroke(u16 Color)
{
  Color_Text = Color;
}

/*********************************************************************************************************//**
 * @brief write RAM Word
 * @param RGB_Set : Color
 * @retval void
 ************************************************************************************************************/
void TFT_writeRAMWord(u32 RGB_Set)
{
  TFT_WriteRAMPrior();

  TFT_WriteRAM(RGB_Set);
}

/*********************************************************************************************************//**
 * @brief Set starting coordinates
 * @param X_Location    : The horizontal position
          Y_Location    : The vertical position
 * @retval void
 ************************************************************************************************************/
void TFT_starterSet(u32 X_Location, u32 Y_Location)
{
  TFT_setAddrWindow(X_Location, Y_Location, 1, 1);
}

/*********************************************************************************************************//**
 * @brief set Text Size
 * @param s : Text Size
 * @retval void
 * @other range from 1 to 255.The larger the number, the larger the font.
 ************************************************************************************************************/
void TFT_setTextSize(uint8_t s)
{ 
	textsize = (s > 0) ? s : 1; 
}

/*********************************************************************************************************//**
 * @brief Draw a horizontal or vertical line
 * @param X_Location    : The horizontal position Of The Starting point
          Y_Location    : The vertical position Of The Starting point
          Length        : Pixel length
          Direction     : Direction
                           @arg Horizontal: 0x00
                           @arg Vertical: 0x01
          Color         : Color
 * @retval void
 ************************************************************************************************************/
void TFT_drawLine(u32 X_Location, u32 Y_Location, u32 Length, u32 Direction,u16 Color)
{
  s32 i = 0;

  if (Direction == Horizontal)
  {
    TFT_setAddrWindow(X_Location, Y_Location, Length, 1);
    for (i = Length; i > 0; i--)
    {
      TFT_WriteRAM(Color);
    }
  }
  else
  {
    TFT_setAddrWindow(X_Location, Y_Location, 1, Length);
    for (i = Length; i > 0; i--)
    {
      TFT_WriteRAM(Color);
    }
  }
}

/*********************************************************************************************************//**
 * @brief Draw a line
 * @param X_Start    : The horizontal position Of The Starting point
          Y_Start    : The vertical position Of The Starting point
          X_End      : The horizontal position Of The Ending point
          Y_End      : The vertical position Of The Ending point
 * @retval void
 * @other  Draw a line from (x1, y1) to (x2, y2)
           Line color: set by the stroke function
 ************************************************************************************************************/
void TFT_line(u32 X_Start, u32 Y_Start, u32 X_End, u32 Y_End)
{
	u32 dx, dy;
	int16_t steep =  0;
	int16_t err = 0;
	int16_t ystep = 0;
	u32 tem = 0;
	if(X_Start == X_End)
	{
		if(Y_Start < Y_End)
		{
			TFT_drawLine(X_Start,Y_Start,(Y_End-Y_Start),Vertical,Color_Text);
		}
		else
		{
			TFT_drawLine(X_Start,Y_End,(Y_Start-Y_End),Vertical,Color_Text);
		}
	}
	else if(Y_Start == Y_End)
	{
		if(X_Start < X_End)
		{
			TFT_drawLine(X_Start,Y_Start,(X_End-X_Start),Horizontal,Color_Text);
		}
		else
		{
			TFT_drawLine(X_End,Y_Start,(X_Start-X_End),Horizontal,Color_Text);
		}
	}
	else
	{
			steep = abs((int)Y_End - (int)Y_Start) > abs((int)X_End - (int)X_Start);
			if (steep) {
				//swap(X_Start, Y_Start);
				tem = X_Start;
				X_Start = Y_Start;
				Y_Start = tem;
				
				//swap(X_End, Y_End);
				tem = X_End;
				X_End = Y_End;
				Y_End = tem;
			}

			if (X_Start > X_End) {
				//swap(X_Start, X_End);
				tem = X_Start;
				X_Start = X_End;
				X_End = tem;
				
				//swap(Y_Start, Y_End);
				tem = Y_Start;
				Y_Start = Y_End;
				Y_End = tem;
			}

			
			dx = X_End - X_Start;
			dy = abs((int)Y_End - (int)Y_Start);

			err = dx / 2;

			if (Y_Start < Y_End) {
				ystep = 1;
			} else {
				ystep = -1;
			}

			for (; X_Start<=X_End; X_Start++) {
				if (steep) {
					TFT_drawPoint(Y_Start, X_Start, Color_Text);
				} else {
					TFT_drawPoint(X_Start, Y_Start, Color_Text);
				}
				err -= dy;
				if (err < 0) {
					Y_Start += ystep;
					err += dx;
				}
			}
	}
}

/*********************************************************************************************************//**
 * @brief fill a Circle
 * @param X_Location    : The horizontal position Of The Circle
          Y_Location    : The vertical position Of The Circle
          Radius        : radius of a circle
          Color         : Color of filling
 * @retval void
 ************************************************************************************************************/
void TFT_fillCircle(u32 X_Location, u32 Y_Location, u32 Radius,u16 Color)
{
  int16_t f     = 1 - Radius;
  int16_t ddF_x = 1;
  int16_t ddF_y = -2 * (int16_t)Radius;
  int16_t x     = 0;
  int16_t y     = Radius;
	uint8_t cornername = 3;
	int16_t delta = 0;
	
	TFT_drawLine(X_Location, Y_Location-Radius,2*Radius+1,Vertical,Color);
	
  while (x<y) {
    if (f >= 0) {
      y--;
      ddF_y += 2;
      f     += ddF_y;
    }
    x++;
    ddF_x += 2;
    f     += ddF_x;

    if (cornername & 0x1) {
			TFT_drawLine(X_Location+x, Y_Location-y,2*y+1+delta,Vertical,Color);
			TFT_drawLine(X_Location+y, Y_Location-x,2*x+1+delta,Vertical,Color);
    }
    if (cornername & 0x2) {
			TFT_drawLine(X_Location-x, Y_Location-y,2*y+1+delta,Vertical,Color);
			TFT_drawLine(X_Location-y, Y_Location-x,2*x+1+delta,Vertical,Color);
    }
  }
}

/*********************************************************************************************************//**
 * @brief draw a Circle
 * @param X_Location    : The horizontal position Of The Circle
          Y_Location    : The vertical position Of The Circle
          Radius        : radius of a circle
          Color         : Color
 * @retval void
 ************************************************************************************************************/
void TFT_drawCircle(u32 X_Location, u32 Y_Location, u32 Radius,u16 Color)
{
  s32  DV;  
  u32  X_Value;      
  u32  Y_Value;      

  DV = 3 - (Radius << 1);
  X_Value = 0;
  Y_Value = Radius;

  while (X_Value <= Y_Value)
  {
    
    TFT_starterSet(X_Location + X_Value, Y_Location + Y_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location + X_Value, Y_Location - Y_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location - X_Value, Y_Location + Y_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location - X_Value, Y_Location - Y_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location + Y_Value, Y_Location + X_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location + Y_Value, Y_Location - X_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location - Y_Value, Y_Location + X_Value);
    TFT_writeRAMWord(Color);

    TFT_starterSet(X_Location - Y_Value, Y_Location - X_Value);
    TFT_writeRAMWord(Color);

    if (DV < 0)
    {
      DV += (X_Value << 2) + 6;
    }
    else
    {
      DV += ((X_Value - Y_Value) << 2) + 10;
      Y_Value--;
    }
    X_Value++;
  }
}

/*********************************************************************************************************//**
 * @brief draw a Circle
 * @param X_Location    : The horizontal position Of The Circle
          Y_Location    : The vertical position Of The Circle
          Radius        : radius of a circle
 * @retval void
 * @other  Circle color: set by the stroke function
           Circle interior color: set by the fill function
 ************************************************************************************************************/
void TFT_circle(u32 X_Location, u32 Y_Location, u32 Radius)
{
	TFT_fillCircle(X_Location,Y_Location,Radius,Color_Back);
	TFT_drawCircle(X_Location,Y_Location,Radius,Color_Text);
	
}

/*********************************************************************************************************//**
 * @brief draw a Horizontal Line
 * @param X_Start  : The horizontal position Of The Starting point
          X_End    : The vertical position Of The Ending point
          Y        : The width of the line
          Color    : Color
 * @retval void
 ************************************************************************************************************/
void TFT_HorizontalLineBorder(s32 X_Start, s32 X_End, s32 Y,u16 Color)
{
	s32 x = 0;

  if (Y < 0 || Y >= _TFT_height) return;
  
  // X_Start <= X_End
  if (X_Start > X_End) {
    s32 temp = X_Start;
    X_Start = X_End;
    X_End = temp;
  }
  

  if (X_Start < 0) X_Start = 0;
  if (X_End >= _TFT_width) X_End = _TFT_width - 1;
  

  if (X_Start > X_End) return;
  

  for (x = X_Start; x <= X_End; x++) {
    TFT_starterSet(x, Y);
    TFT_writeRAMWord(Color);
  }
}

/*********************************************************************************************************//**
 * @brief Solid Circle
 * @param X_Location  : The horizontal position Of The Circle
          Y_Location  : The vertical position Of The Circle
          Radius      : radius of a circle
          Xstart      : The horizontal position Of The Starting Point of the restricted area
          Xend        : The horizontal position Of The Ending Point of the restricted area
          Ystart      : The vertical position Of The Starting Point of the restricted area
          Yend        : The vertical position Of The Ending Point of the restricted area
          Color       : Color
 * @retval void
 ************************************************************************************************************/
void TFT_SolidCircleDrawBorder(s32 X_Location, s32 Y_Location, u32 Radius,u16 Xstart,u16 Xend,u16 Ystart,u16 Yend,u16 Color)
{
  s32  DV;       // Decision Variable
  s32  X_Value;  // Current X Value
  s32  Y_Value;  // Current Y Value


  if (X_Location + (s32)Radius < Xstart || X_Location - (s32)Radius > Xend ||
      Y_Location + (s32)Radius < Ystart || Y_Location - (s32)Radius > Yend) {
    return;
  }

  DV = 3 - (s32)(Radius << 1);
  X_Value = 0;
  Y_Value = (s32)Radius;

  while (X_Value <= Y_Value)
  {
    
    #define CLIP_X(x) ((x) < Xstart ? Xstart : ((x) > Xend ? Xend : (x)))
    #define CLIP_Y(y) ((y) < Ystart ? Ystart : ((y) > Yend ? Yend : (y)))

    s32 y_top = CLIP_Y(Y_Location + Y_Value);
    s32 y_bottom = CLIP_Y(Y_Location - Y_Value);
    s32 y_horiz1 = CLIP_Y(Y_Location + X_Value);
    s32 y_horiz2 = CLIP_Y(Y_Location - X_Value);
    
     if(y_top >= Ystart && y_top <= Yend) {
      s32 x1 = CLIP_X(X_Location - X_Value);
      s32 x2 = CLIP_X(X_Location + X_Value);
      if(x1 <= x2) TFT_HorizontalLineBorder(x1, x2, y_top, Color);
    }
    
    if(y_bottom >= Ystart && y_bottom <= Yend) {
      s32 x1 = CLIP_X(X_Location - X_Value);
      s32 x2 = CLIP_X(X_Location + X_Value);
      if(x1 <= x2) TFT_HorizontalLineBorder(x1, x2, y_bottom, Color);
    }

    if(y_horiz1 >= Ystart && y_horiz1 <= Yend) {
      s32 x1 = CLIP_X(X_Location - Y_Value);
      s32 x2 = CLIP_X(X_Location + Y_Value);
      if(x1 <= x2) TFT_HorizontalLineBorder(x1, x2, y_horiz1, Color);
    }

    if(y_horiz2 >= Ystart && y_horiz2 <= Yend) {
      s32 x1 = CLIP_X(X_Location - Y_Value);
      s32 x2 = CLIP_X(X_Location + Y_Value);
      if(x1 <= x2) TFT_HorizontalLineBorder(x1, x2, y_horiz2, Color);
    }
    
    if (DV < 0) {
      DV += (X_Value << 2) + 6;
    } else {
      DV += ((X_Value - Y_Value) << 2) + 10;
      Y_Value--;
    }
    X_Value++;
  }
}

/*********************************************************************************************************//**
 * @brief draw a Rect
 * @param X_Location  : The horizontal position Of The Starting Point
          Y_Location  : The vertical position Of The Starting Point
          Width       : Width of rectangle
          Height      : Height of rectangle
          Color       : Color
 * @retval void
 ************************************************************************************************************/
void TFT_drawRect(u32 X_Location, u32 Y_Location, u32 Width, u32 Height,u16 Color)
{
  TFT_drawLine(X_Location, Y_Location, Width, Horizontal,Color);
  TFT_drawLine(X_Location, (Y_Location+ Height-1), Width, Horizontal,Color);

  TFT_drawLine(X_Location, Y_Location, Height, Vertical,Color);
  TFT_drawLine(X_Location + Width - 1, Y_Location, Height, Vertical,Color);
}

/*********************************************************************************************************//**
 * @brief draw a Rect
 * @param X_Location  : The horizontal position Of The Starting Point
          Y_Location  : The vertical position Of The Starting Point
          Width       : Width of rectangle
          Height      : Height of rectangle
 * @retval void
 * @other Rectangle color: set by the stroke function
           Rectangle interior color: set by the fill function
 ************************************************************************************************************/
void TFT_rect(u16 Xstart, u16 Ystart, u16 Width, u16 Height)
{
	TFT_fillRect(Xstart,Ystart,Width,Height,Color_Back);
	TFT_drawRect(Xstart,Ystart,Width,Height,Color_Text);
}

/*********************************************************************************************************//**
 * @brief draw Char(ASCII_Font_Table)
 * @param X_Location  : The horizontal position Of The Starting Point
          Y_Location  : The vertical position Of The Starting Point
          *Cptr       : ASCII_Font_Table
 * @retval void
 * @other Font Format: Font2
 ************************************************************************************************************/
void TFT_drawChar_Font2(u32 X_Location, u32 Y_Location, u16 *Cptr)
{
  u32 i = 0, j = 0;

  for (i = 0; i < Font_Raw; i++)
  {
    TFT_setAddrWindow(X_Location, Y_Location++, Font_Column, 1);
    for (j = 0; j < Font_Column; j++)
    {
      if ((Cptr[i] & (1 << j)) == 0x00)
      {
        TFT_WriteRAM(Color_Back);
      }
      else
      {
        TFT_WriteRAM(Color_Text);
      }
    }
  }
}

/*********************************************************************************************************//**
 * @brief display Char(Ascii)
 * @param X_Location  : The horizontal position Of The Starting Point
          Y_Location  : The vertical position Of The Starting Point
          Ascii       : Want to display ASCII code of characters
 * @retval void
 * @other Font Format: Font2
 ************************************************************************************************************/
void TFT_displayChar_Font2(u32 X_Location, u32 Y_Location, u32 Ascii)
{
  Ascii -= 32;
  TFT_drawChar_Font2(X_Location, Y_Location, (u16 *)&ASCII_Font_Table2[Ascii * 24]);
}

/*********************************************************************************************************//**
 * @brief display String(x,y)
 * @param X_Location  : The horizontal position Of The Starting Point
          Y_Location  : The vertical position Of The Starting Point
          *Sptr       : The string content you want to display
 * @retval void
 * @other Font Format: Font2
 ************************************************************************************************************/
void TFT_displayString_Font2(u32 X_Location, u32 Y_Location, char *Sptr)
{
  u32 i = 0;
  u16 reference_column = X_Location;

  /* Send character by character on LCD                                                                     */
  while ((*Sptr != 0) & (i < 20))
  {
    TFT_displayChar_Font2(reference_column, Y_Location, *Sptr);
    /* Increment the column position by 16, because character is size of 16x24                              */
    reference_column += 16;
    /* Point to the next character                                                                          */
    Sptr++;

    /* Increase the character counter                                                                       */
    i++;
  }
}

/*********************************************************************************************************//**
 * @brief Display a line of string(x=0,Line_Num)
 * @param Line_Num  : Num of the Line
          *Sptr     : The string content you want to display
 * @retval void
 * @other Font Format: Font2
 ************************************************************************************************************/
void TFT_displayStringLine_Font2(u32 Line_Num, char *Sptr)
{
  u32 i = 0;
  u16 reference_column = 0;

  /* Send character by character on LCD                                                                     */
  while ((*Sptr != 0) & (i < 20))
  {
    TFT_displayChar_Font2(reference_column, Line_Num, *Sptr);
    /* Increment the column position by 16, because character is size of 16x24                              */
    reference_column += 16;
    /* Point to the next character                                                                          */
    Sptr++;

    /* Increase the character counter                                                                       */
    i++;
  }
}

/*********************************************************************************************************//**
 * @brief draw Char
 * @param x     : The horizontal position Of The Starting Point
          y     : The vertical position Of The Starting Point
          c     : The characters you want to display , eg:'A'
          color : Color of characters
          bg    : Background color of characters
          size  : Character size
 * @retval void
 * @other Font Format: Font1(Similar Arduino Library)
 ************************************************************************************************************/
void TFT_drawChar(int16_t x, int16_t y, unsigned char c,uint16_t color, uint16_t bg, uint8_t size)
{
	int8_t i=0;
	int8_t j = 0;
  if((x >= _TFT_width)    || // Clip right
     (y >= _TFT_height)    || // Clip bottom
     ((x + 6 * size - 1) < 0)  || // Clip left
     ((y + 8 * size - 1) < 0))     // Clip top
    return;

  for (i=0; i<6; i++ ) {
    uint8_t line;
    if (i == 5)
      line = 0x0;
    else
      line = pgm_read_byte(gFont+(c*5)+i);
    for (j = 0; j<8; j++) {
      if (line & 0x1) {
        if (size == 1) // default size
          TFT_drawPoint(x+i, y+j, color);
        else {  // big size
          TFT_fillRect(x+(i*size), y+(j*size), size, size, color);
        }
      } else if (bg != color) {
        if (size == 1) // default size
          TFT_drawPoint(x+i, y+j, bg);
        else {  // big size
          TFT_fillRect(x+i*size, y+j*size, size, size, bg);
        }
      }
      line >>= 1;
    }
  }
}

/*********************************************************************************************************//**
 * @brief display String(x,y)
 * @param *text : The string content you want to display
          x     : The horizontal position Of The Starting Point
          y     : The vertical position Of The Starting Point
 * @retval void
 * @other Font Format: Font1(Similar Arduino Library)
          Character size: set by the setTextSize function
          Character color: set by the stroke function
 ************************************************************************************************************/
void TFT_text(const char *text, int16_t x, int16_t y)
{
  while(*text!='\0')
  {       
    TFT_drawChar(x,y,*text,Color_Text,Color_Back,textsize);
    x=x+textsize*5+1;
    text++;
  }  
}

/*********************************************************************************************************//**
 * @brief TFT Config
 * @param void
 * @retval void
 ************************************************************************************************************/
static void TFT_Config(void)
{
    delay_ms(100);

    TFT_WriteReg8(0xEF );
    TFT_WriteRAM8(0x03 );
    TFT_WriteRAM8(0x80 );
    TFT_WriteRAM8(0x02 );
    
    TFT_WriteReg8(0xCF );
    TFT_WriteRAM8(0x00 );
    TFT_WriteRAM8(0xC1 ); 
    TFT_WriteRAM8(0x30 ); 
    
    TFT_WriteReg8(0xED );
    TFT_WriteRAM8(0x64 ); 
    TFT_WriteRAM8(0x03 ); 
    TFT_WriteRAM8(0x12 );
    TFT_WriteRAM8(0x81 ); 
    
    TFT_WriteReg8(0xE8 );
    TFT_WriteRAM8(0x85 ); 
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x78 ); 
    
    TFT_WriteReg8(0xCB );
    TFT_WriteRAM8(0x39 ); 
    TFT_WriteRAM8(0x2C ); 
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x34 ); 
    TFT_WriteRAM8(0x02 );      
    
    TFT_WriteReg8(0xF7 );
    TFT_WriteRAM8(0x20 ); 
    
    TFT_WriteReg8(0xEA );
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x00 ); 
    
    TFT_WriteReg8(0xC0 );
    TFT_WriteRAM8(0x23 ); 
    
    TFT_WriteReg8(0xC1 );
    TFT_WriteRAM8(0x10 ); 
    
    TFT_WriteReg8(0xC5 );
    TFT_WriteRAM8(0x3E ); 
    TFT_WriteRAM8(0x28 ); 
    
    TFT_WriteReg8(0xC7 );
    TFT_WriteRAM8(0x86 ); 
    
    TFT_WriteReg8(0x36 );
    TFT_WriteRAM8(0x48 ); 
    
    TFT_WriteReg8(0x37 );
    TFT_WriteRAM8(0x00 );   
    
    TFT_WriteReg8(0x3A );
    TFT_WriteRAM8(0x55 ); 
    
    TFT_WriteReg8(0xB1 );
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x1F ); //0x18
    
    TFT_WriteReg8(0xB6 );
    TFT_WriteRAM8(0x08 ); 
    TFT_WriteRAM8(0x82 ); 
    TFT_WriteRAM8(0x27 ); 
    
    TFT_WriteReg8(0xF2 );        
    TFT_WriteRAM8(0x00 ); 
    
    TFT_WriteReg8(0x26 );
    TFT_WriteRAM8(0x01 ); 
    
    TFT_WriteReg8(0xE0 );
    TFT_WriteRAM8(0x0F ); 
    TFT_WriteRAM8(0x31 ); 
    TFT_WriteRAM8(0x2B ); 
    TFT_WriteRAM8(0x0C ); 
    TFT_WriteRAM8(0x0E ); 
    TFT_WriteRAM8(0x08 ); 
    TFT_WriteRAM8(0x4E ); 
    TFT_WriteRAM8(0xF1 ); 
    TFT_WriteRAM8(0x37 ); 
    TFT_WriteRAM8(0x07 ); 
    TFT_WriteRAM8(0x10 ); 
    TFT_WriteRAM8(0x03 ); 
    TFT_WriteRAM8(0x0E ); 
    TFT_WriteRAM8(0x09 ); 
    TFT_WriteRAM8(0x00 ); 
    
    TFT_WriteReg8(0xE1 );
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x0E ); 
    TFT_WriteRAM8(0x14 ); 
    TFT_WriteRAM8(0x03 ); 
    TFT_WriteRAM8(0x11 ); 
    TFT_WriteRAM8(0x07 ); 
    TFT_WriteRAM8(0x31 ); 
    TFT_WriteRAM8(0xC1 ); 
    TFT_WriteRAM8(0x48 ); 
    TFT_WriteRAM8(0x08 ); 
    TFT_WriteRAM8(0x0F ); 
    TFT_WriteRAM8(0x0C ); 
    TFT_WriteRAM8(0x31 ); 
    TFT_WriteRAM8(0x36 ); 
    TFT_WriteRAM8(0x0F ); 
    
    TFT_WriteReg8(0x11 );
    delay_ms(150);
    
    TFT_WriteReg8(0x36 );
    TFT_WriteRAM8(MAC ); 
    
    TFT_WriteReg8(0x29 );
    
    TFT_WriteReg8(0x35 );
    TFT_WriteRAM8(0x00 ); 
    
    TFT_WriteReg8(0x44 );
    TFT_WriteRAM8(0x00 ); 
    TFT_WriteRAM8(0x04 ); 


}


/*********************************************************************************************************//**
 * @brief TFT Configuration
 * @param void
 * @retval void
 ************************************************************************************************************/
void TFT_Configuration(void)
{
  CKCU_PeripClockConfig_TypeDef CKCUClock = {{ 0 }};
  CKCUClock.Bit.AFIO              = 1;
  CKCUClock.Bit.PC     = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);

#if (LCD_MODE==LCD_SPI_MODE) 
  AFIO_GPxConfig(LCD_SPI_DC_GPIO_ID, LCD_SPI_DC_GPIO_PIN, AFIO_FUN_GPIO);
  LCD_DC_HIGH;
  GPIO_DirectionConfig(LCD_SPI_DC_GPIO_PORT, LCD_SPI_DC_GPIO_PIN, GPIO_DIR_OUT);

  AFIO_GPxConfig(LCD_SPI_BL_GPIO_ID, LCD_SPI_BL_GPIO_PIN, AFIO_FUN_GPIO);
  LCD_BL_HIGH;
  GPIO_DirectionConfig(LCD_SPI_BL_GPIO_PORT, LCD_SPI_BL_GPIO_PIN, GPIO_DIR_OUT);
  
#endif   
  
  //Please note that TFT_setRotation needs to be set before the TFT_Config
  //TFT_setRotation(1);
  TFT_Config();
  
  delay_ms(20);
  
}

/*********************************************************************************************************//**
 * @brief Draw pixels within an area(draw picture)
 * @param X_Location : The horizontal position Of The Starting Point
          Y_Location : The vertical position Of The Starting Point
          Width      : The Width Of area
          Height     : The Height Of area
          *Pptr      : Array(point to pixel data of picture.)
 * @retval void
 ************************************************************************************************************/
void TFT_drawPixels(u8 X_Location, u16 Y_Location, u16 Width, u8 Height, uc8 *Pptr)
{
  u32 xid = 0;
  u32 ImgAdds = 0;
  u32 yid = 0;
  u32 i = 0, j = 0, color = 0;

  xid = X_Location;
  yid = Y_Location;
  TFT_setAddrWindow(xid,yid,Width,Height);
  for (i = 0; i < Height; i++)
  {
    for (j = 0; j < Width; j++)
    {
      ImgAdds = (i * Width * 2) + (j * 2);
      color = Pptr[ImgAdds] << 8 | (Pptr[ImgAdds + 1]);
      TFT_WriteRAM(color);
    }
  }
}


/*********************************************************************************************************//**
  * @brief  Set the display direction.
  * @param  dir: display direction.
              0:  0 degrees
              1:  90 degrees
              2:  180 degrees
              3:  270 degrees
  * @retval void
  ***********************************************************************************************************/
void TFT_setRotation(u8 dir)
{
  
  if(dir==1){
    _TFT_width= 320;
    _TFT_height=  240;
    MAC= 0x28;
    TFT_DIRECTION=1;
  }
  else if(dir==2){
  _TFT_width=   240;
  _TFT_height=  320;
  MAC=         0x88;
    TFT_DIRECTION=2;
  }
  else if(dir==3){
  _TFT_width=   320;
  _TFT_height=  240;
  MAC=         0xE8;
    TFT_DIRECTION=3;
  }
  else if(dir==0)
  {
    _TFT_width=   240;
    _TFT_height=  320;
    MAC=         0x48;
    TFT_DIRECTION=0;
  }
   // TFT_WriteReg(0x36,MAC); //Set in  TFT_Configuration
  
}

/*********************************************************************************************************//**
  * @brief  Get the display direction.
  * @param  void
  * @retval dir: display direction.
              0:  0 degrees
              1:  90 degrees
              2:  180 degrees
              3:  270 degrees
  ***********************************************************************************************************/
u8 TFT_getRotation(void)
{
	return TFT_DIRECTION;
}

/*********************************************************************************************************//**
  * @brief  Get the height of TFT.
  * @param  void
  * @retval height
  ***********************************************************************************************************/
int16_t TFT_height(void)
{
	return _TFT_height;
}

/*********************************************************************************************************//**
  * @brief  Get the width of TFT.
  * @param  void
  * @retval width
  ***********************************************************************************************************/
int16_t TFT_width(void)
{
	return _TFT_width;
}




