/*********************************************************************************************************//**
 * @file    BMD58T280_HT32/src/XPT2046.c
 * @version V1.0.1
 * @date    2025-07-18
 * @brief   The function of XPT2046 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
 //-----------------------------------------------------------------------------
#include "XPT2046.h"
#include "TFT_driver_ILI9341.h"
#include "delay.h"
#include <math.h>
#include "clear32X32top2btm.h"
#include "SPI.h"

//-----------------------------------------------------------------------------

u8 cmd_rdx = 0xd0;
u8 cmd_rdy = 0x90;

u16 Xdown=0xFFFF; 
u16 Ydown=0xFFFF;
u16 Xup=0;
u16 Yup=0; 

u16 x=0XFFFF;
u16 y=0XFFFF;
u8 time;


#define minXValue       164.9
#define maxXValue       3762.5

#define minYValue       197.4
#define maxYValue       3707.1


//The following parameters are the default values and can be adjusted according to the situation of each LCD
float xFactor=0.06671114;
float yFactor=0.09117551;
short xOffset=-11;      
short yOffset=-18;

extern u16 _TFT_width;
extern u16 _TFT_height;
extern u16 MAC;
extern u8 TFT_DIRECTION;



/*********************************************************************************************************//**
  * @brief  XPT2046 Configuration.
  * @param  void
  * @retval void
  ***********************************************************************************************************/
void XPT2046_Configuration(void)
{

  CKCU_PeripClockConfig_TypeDef CKCUClock = {{ 0 }};
  CKCUClock.Bit.PC    = 1;
  CKCUClock.Bit.AFIO  = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);

  //PEN
  GPIO_DirectionConfig    (XPT2046_PEN_GPIO_PORT, XPT2046_PEN_GPIO_PIN, GPIO_DIR_IN);
  GPIO_InputConfig        (XPT2046_PEN_GPIO_PORT, XPT2046_PEN_GPIO_PIN, ENABLE);
  GPIO_PullResistorConfig (XPT2046_PEN_GPIO_PORT, XPT2046_PEN_GPIO_PIN, GPIO_PR_UP);
  
}

/*********************************************************************************************************//**
  * @brief  this function is read data from touch.
  * @param  *x : x coordinate vaule.
            *y : y coordinate vaule.
  * @retval status
             @arg 1:Sucess
             @arg 0:False
  ***********************************************************************************************************/
u8 touch_read_xy(u16 *x, u16 *y)
{
  u16 xtemp, ytemp;
  xtemp = touch_read_x_or_y(cmd_rdx);
  ytemp = touch_read_x_or_y(cmd_rdy);
  if(xtemp<50||ytemp<50)return 0;
  
  *x = xtemp;
  *y = ytemp;
  
  return 1;
}

/*********************************************************************************************************//**
  * @brief  this function is read data from touch(Average of two touches)
  * @param  *x : x coordinate vaule(Average of two touches)
            *y : y coordinate vaule(Average of two touches)
  * @retval status
             @arg 1:Sucess
             @arg 0:False
  ***********************************************************************************************************/
u8 touch_read_xy2(u16 *x,u16 *y) 
{
  u16 x1,y1;
  u16 x2,y2;
  u8 flag;    
  flag=touch_read_xy(&x1,&y1);   
  if(flag==0)return(0);
  flag=touch_read_xy(&x2,&y2);	   
  if(flag==0)return(0);   
  if(((x2<=x1&&x1<x2+50)||(x1<=x2&&x2<x1+50)) 
  &&((y2<=y1&&y1<y2+50)||(y1<=y2&&y2<y1+50)))
  {
      *x=(x1+x2)/2;
      *y=(y1+y2)/2;
      return 1;
  }else return 0;	  
}


/*********************************************************************************************************//**
  * @brief  this function is read data from touch(read five times, remove max and min, and then take the average)
  * @param  xy : x or y data
  * @retval processed data
  ***********************************************************************************************************/
#define READ_TIMES                       5
#define LOST_VAL                         1
u16 touch_read_x_or_y(u8 xy)
{
  u16 i, j;
  u16 buf[READ_TIMES];
  u16 sum = 0;
  u16 temp;

  for(i = 0; i < READ_TIMES; i++)
  {
    buf[i] = touch_read_ad(xy);
  }

  for(i = 0; i < READ_TIMES - 1; i++)
  {
    for(j = i + 1; j < READ_TIMES; j++)
    {
      if(buf[i] > buf[j])
      {
        temp = buf[i];
        buf[i] = buf[j];
        buf[j] = temp;
      }
    }
  }

  sum = 0;

  for(i = LOST_VAL; i < READ_TIMES - LOST_VAL; i++)
  {
    sum += buf[i];
  }

  temp = sum/(READ_TIMES - 2 * LOST_VAL);

  return temp;
}

/*********************************************************************************************************//**
  * @brief  this function is read data from touch(read one time)
  * @param  xy : x or y data
  * @retval data
  ***********************************************************************************************************/
u16 touch_read_ad(u8 xy)
{
  uint16_t num = 0;  
  SPI1_SwitchToLowSpeed();
  TP_CS_LOW;
  SPI1_TP_WriteByte(xy);
  delay_us(6);
  num = SPI1_TP_ReadByte();
  num = num << 8;
  num |= SPI1_TP_ReadByte();
  num >>= 3;
  TP_CS_HIGH;
  SPI1_SwitchToHighSpeed();
  
  return(num);
}

/*********************************************************************************************************//**
  * @brief  this function is scanning the screen.
  * @param  void
  * @retval void
  ***********************************************************************************************************/
void touch_scan(void)
{
  u16 x1,y1;
  Xup=0xffff;
  Yup=0xffff;

  if(TP_IRQ_STATE == RESET)
  { 
    if(touch_read_xy2(&x,&y))
    {
      if(TFT_DIRECTION == 0){
        x1=x;
        y1=y;
        x=((xFactor*x1+xOffset)<0)?0:xFactor*x1+xOffset;    
        y=((yFactor*y1+yOffset)<0)?0:yFactor*y1+yOffset;
      }
      
      /* 
      
      After modifying the display direction, it is necessary to synchronously modify the corresponding display font coordinates, image coordinates, and touch range coordinates
      
      */
      else if(TFT_DIRECTION == 1){
        x1=y;
        y1=minYValue+maxYValue-x;
        x=((yFactor*x1+yOffset)<0)?0:yFactor*x1+yOffset;    
        y=((xFactor*y1+xOffset)<0)?0:xFactor*y1+xOffset;
      }
      else if(TFT_DIRECTION == 2){
        x1=minXValue+maxXValue-x;
        y1=minYValue+maxYValue-y;
        x=((xFactor*x1+xOffset)<0)?0:xFactor*x1+xOffset;    
        y=((yFactor*y1+yOffset)<0)?0:yFactor*y1+yOffset;
      }
      else if(TFT_DIRECTION == 3){
        x1=minYValue+maxYValue-y;
        y1=x;
        x=((yFactor*x1+yOffset)<0)?0:yFactor*x1+yOffset;    
        y=((xFactor*y1+xOffset)<0)?0:xFactor*y1+xOffset;
      }
      
    }
		
    Xdown=x;
    Ydown=y;
		
    time++;		   
  }
	else  
  {    
     if(time>2)
     {
          Xup=x;
          Yup=y;	 
     }
     time=0;
     Xdown=0xffff;
     Ydown=0xffff;	 
  }
      

}

/*********************************************************************************************************//**
  * @brief  Example1: Painting(Return touch coordinates)
  * @param  void
  * @retval void
  ***********************************************************************************************************/
void T_test(void)
{
  TFT_background(White);
    
  TFT_displayStringLine_Font2(Line1, "Please Touch");
  TFT_displayStringLine_Font2(Line2, "the screen");

  while(1)
  {
    touch_scan(); 		 

    if(Xdown<_TFT_width&&Ydown<_TFT_height)
    {
      char showStr[20];
      sprintf(showStr,"(%d,%d)",Xdown,Ydown);
      TFT_fillRect(0, 96, _TFT_width, 24, White);
      TFT_displayStringLine_Font2(Line4, showStr);
    } 
  }	
}

/*********************************************************************************************************//**
  * @brief  //Example2: Painting
  * @param  void
  * @retval void
  ***********************************************************************************************************/
void T_test2(void)
{
	u16 tempColor=Yellow;
  TFT_background(White);
  TFT_drawPixels((picX), (picY), picW, picH, gImage_clear32X32top2btm);
  TFT_fillRect(picX+aim_margin+picW,picY,picH,picW,Yellow);
  TFT_fillRect(picX+aim_margin*2+picW*2,picY,picH,picW,Green);
  TFT_drawLine(0,picY+picH+aim_margin,_TFT_width,Horizontal,Yellow);

		while(1)
		{
      touch_scan(); 		 

			if(Xdown<_TFT_width&&Ydown<_TFT_height)
			{
        
        if(Ydown>picY+picH+aim_margin&&Ydown<_TFT_height)
            {
              
              TFT_SolidCircleDrawBorder(Xdown, Ydown,circle_Radius,0,_TFT_width,picY+picH+aim_margin,_TFT_height,tempColor);  
            }
            else if(Xdown>picX+aim_margin+picW&&Xdown<picX+aim_margin+picW+picW&&Ydown>picY-picH+YellowYoffset&&Ydown<picY+YellowYoffset)
            { 
              
               //PicYoffset and YellowYoffset may vary in each LCD, and can be adjusted if necessary
               tempColor = Yellow;
            }
            else if(Xdown>picX&&Xdown<picX+picW&&Ydown>picY-picH+picYoffset&&Ydown<picY+picYoffset)
            { 
              //PicYoffset and YellowYoffset may vary in each LCD, and can be adjusted if necessary
              TFT_drawLine(0,picY+picH+aim_margin,_TFT_width,Horizontal,Yellow);
              TFT_fillRect(0,picY+picH+aim_margin+1,_TFT_width,_TFT_height-(picY+picH+aim_margin),White);
            }  
            else if(Xdown>picX+aim_margin*2+picW*2&&Xdown<picX+aim_margin*2+picW*2+picW&&Ydown>picY&&Ydown<picY+picH)
            {
              tempColor=Green;
            }
			} 
		}	
}







