/*****************************************************************
File:           BM22S4421-1.cpp
Author:         BESTMODULES
Description:    None
Version:          V1.0.1    -- 2023-12-4
******************************************************************/
#include "BM22S4421-1.h"

/**********************************************************
Description: Constructor
Parameters: statusPin: Status pin connection with Arduino or BMduino
            *theSerial: Serial object, if your board has multiple UART interfaces
Return: None
Others: None
**********************************************************/
BM22S4421_1::BM22S4421_1(uint8_t statusPin, HardwareSerial *theSerial)
{
  _softSerial = NULL;
  _statusPin = statusPin;
  _hardSerial = theSerial;
}

/**********************************************************
Description:  Constructor
Parameters: statusPin: Status pin connection with Arduino or BMduino
            rxPin : Receiver pin of the UART
            txPin : Send signal pin of UART
Return: None
Others: None
**********************************************************/
BM22S4421_1::BM22S4421_1(uint8_t statusPin, uint8_t rxPin, uint8_t txPin)
{
  _hardSerial = NULL;
  _statusPin = statusPin;
  _rxPin = rxPin;
  _txPin = txPin;
  _softSerial = new SoftwareSerial(_rxPin, _txPin);
}

/**********************************************************
Description: Module initial
Parameters: None
Return: None
Others: None
**********************************************************/
void BM22S4421_1::begin()
{
  if (_softSerial != NULL)
  {
    _softSerial->begin(BAUDRATE); // baud rate:4800
  }
  if (_hardSerial != NULL)
  {
    _hardSerial->begin(BAUDRATE); // baud rate:4800
  }
  pinMode(_statusPin, INPUT);
}

/**********************************************************
Description: Get FW version
Parameters: None
Return: FW version
Others: None
**********************************************************/
uint8_t BM22S4421_1::getFWVer()
{
  uint8_t sendBuf[5] = {0x00, 0x43, 0x15, 0x00, 0x00};
  uint8_t recBuf[5] = {0};
  sendBuf[4] = sendBuf[2] + sendBuf[3] + 4;
  writeBytes(sendBuf);
  readBytes(recBuf, 5);
  if (recBuf[2] == 0x15)
  {
    return recBuf[3];
  }
  else
  {
    return 0;
  }
}

/**********************************************************
Description: Set sensing distance
Parameters: value = 1:  5m
            value = 2:  9m
            value = 3: 10m
            value = 4: 11m
            value = 5: 12m
            value = 6: 13m
            value = 7: 14m
Return: 0: set successed
        1: set failed
        2: communication failed
Others: None
**********************************************************/
uint8_t BM22S4421_1::setSensingDistance(uint8_t value)
{
  uint8_t sendBuf[5] = {0x00, 0x42, 0x01, value, 0x00};
  uint8_t recBuf[5] = {0};
  sendBuf[4] = sendBuf[2] + sendBuf[3] + 4;
  writeBytes(sendBuf);
  readBytes(recBuf, 1, 50);
  if (recBuf[0] == 0x6A)
  {
    return CHECK_OK;
  }
  else if (recBuf[0] == 0x65)
  {
    return CHECK_ERROR;
  }
  else
  {
    return TIMEOUT_ERROR;
  }
}

/**********************************************************
Description: Set OUT Pin output mode
Parameters: LOW_LEVEL(1): OUT pin output low level when an object is detected
            HIGH_LEVEL(2): OUT pin output high level when an object is detected
Return: 0: set successed
        1: set failed
        2: communication failed
Others: None
**********************************************************/
uint8_t BM22S4421_1::setOutMode(uint8_t mode)
{
  uint8_t sendBuf[5] = {0x00, 0x42, 0x02, mode, 0x00};
  uint8_t recBuf[5] = {0};
  sendBuf[4] = sendBuf[2] + sendBuf[3] + 4;
  writeBytes(sendBuf);
  readBytes(recBuf,1);
  if (recBuf[0] == 0x6A)
  {
    return CHECK_OK;
  }
  else if (recBuf[0] == 0x65)
  {
    return CHECK_ERROR;
  }
  else
  {
    return TIMEOUT_ERROR;
  }
}

/**********************************************************
Description: Set OUT Pin output time
Parameters: Time of output signal of OUT pin when object is detected,
            the range is 1 to 250. Unit: seconds
Return: 0: set successed
        1: set failed
        2: communication failed
Others: None
**********************************************************/
uint8_t BM22S4421_1::setOutTime(uint8_t time)
{
  uint8_t sendBuf[5] = {0x00, 0x42, 0x03, time, 0x00};
  uint8_t recBuf[5] = {0};
  sendBuf[4] = sendBuf[2] + sendBuf[3] + 4;
  writeBytes(sendBuf);
  readBytes(recBuf, 1);
  if (recBuf[0] == 0x6A)
  {
    return CHECK_OK;
  }
  else if (recBuf[0] == 0x65)
  {
    return CHECK_ERROR;
  }
  else
  {
    return TIMEOUT_ERROR;
  }
}

/**********************************************************
Description: Write data through UART
Parameters: wBuf: The array for storing data to be sent(9 bytes)
            wLen:Length of data sent
Return: None
Others: None
**********************************************************/
void BM22S4421_1::writeBytes(uint8_t wbuf[], uint8_t wLen)
{
  uint8_t i;
  /*Select hardSerial or softSerial according to the setting*/
  if (_softSerial != NULL)
  {
    while (_softSerial->available() > 0)
    {
      _softSerial->read();
    }
    for (i = 0; i < wLen; i++)
    {
      _softSerial->write(wbuf[i]);
      _softSerial->flush(); // Wait for the end of serial port data transmission
      delayMicroseconds(500);
    }
  }
  else
  {
    while (_hardSerial->available() > 0)
    {
      _hardSerial->read();
    }
    for (i = 0; i < wLen; i++)
    {
      _hardSerial->write(wbuf[i]);
      _hardSerial->flush(); // Wait for the end of serial port data transmission
      delayMicroseconds(500);
    }
  }
}

/**********************************************************
Description: Read data through UART
Parameters: rBuf: The array for storing Data to be sent
            timeout: Receive timeout(unit: ms)
Return: 0: Read succeeded
        1: Timeout error
Others: None
**********************************************************/
uint8_t BM22S4421_1::readBytes(uint8_t rBuf[], uint8_t rLen, uint16_t timeout)
{
  uint8_t i = 0;
  uint16_t delayCnt = 0;
  /* Select SoftwareSerial Interface */
  if (_softSerial != NULL)
  {
    for (i = 0; i < rLen; i++)
    {
      delayCnt = 0;
      while (_softSerial->available() == 0)
      {
        if (delayCnt > timeout)
        {
          return TIMEOUT_ERROR; // Timeout error
        }
        delay(1);
        delayCnt++;
      }
      rBuf[i] = _softSerial->read();
    }
  }
  /* Select HardwareSerial Interface */
  if (_hardSerial != NULL)
  {
    for (i = 0; i < rLen; i++)
    {
      delayCnt = 0;
      while (_hardSerial->available() == 0)
      {
        if (delayCnt > timeout)
        {
          return TIMEOUT_ERROR; // Timeout error
        }
        delay(1);
        delayCnt++;
      }
      rBuf[i] = _hardSerial->read();
    }
  }
}
