
#include "BMZ00050_I2C.h"
#include "BMZ00050_debug.h"
#include "Arduino.h"

#define BMZ00050_I2C_ADDRESS        (0x50 >> 1)


BMZ00050_I2C::BMZ00050_I2C(TwoWire &wire)
{
    _wire = &wire;
    command = 0;
}

void BMZ00050_I2C::begin()
{
    _wire->begin();
}

int8_t BMZ00050_I2C::writeCommand(const uint8_t *header, uint8_t hlen, const uint8_t *body, uint8_t blen)
{
    command = header[0];
    _wire->beginTransmission(BMZ00050_I2C_ADDRESS);

    write(PREAMBLE);

    write(STARTCODE1);

    write(STARTCODE2);
    
    uint8_t length = hlen + blen + 1;   // length of data field: TFI + DATA
    write(length);
   
    write(~length + 1);                 // checksum of length
   
    write(HOSTTOBMZ00050);
    uint8_t sum = HOSTTOBMZ00050;    // sum of TFI + DATA
   
    DMSG("\nwrite: ");
       
    for (uint8_t i = 0; i < hlen; i++) {
          
        if (write(header[i])) {
            sum += header[i];
            
            DMSG_HEX(header[i]);
        } else {
            DMSG("\nToo many data to send, I2C doesn't support such a big packet\n");     // I2C max packet: 32 bytes
            return BMZ00050_INVALID_FRAME;
        }
    }

    for (uint8_t i = 0; i < blen; i++) {
        if (write(body[i])) {
            sum += body[i];
            
            DMSG_HEX(body[i]);
        } else {
            DMSG("\nToo many data to send, I2C doesn't support such a big packet\n");     // I2C max packet: 32 bytes
            return BMZ00050_INVALID_FRAME;
        }
    }
    uint8_t checksum = ~sum + 1;            // checksum of TFI + DATA
    write(checksum);
    write(POSTAMBLE);  

    _wire->endTransmission();
  
    DMSG('\n');

    return readAckFrame();
}
int16_t BMZ00050_I2C::readResponse(uint8_t buf[], uint8_t len, uint16_t timeout)
{
    uint16_t time = 0;
 
    do {
        if (_wire->requestFrom(BMZ00050_I2C_ADDRESS, len)) 
        {
            if (read() & 1) 
            {                  
                               // check first byte --- IICSR 
                break;         // BMZ00050 is ready
            }
        }

        delay(1);
        time++;
        if ((0 != timeout) && (time > timeout)) {
            return -1;
        }
    } while (1); 
    
    if (0x00 != read()  ||            // PREAMBLE
        0x00 != read()  ||            // STARTCODE1
        0xFF != read()                // STARTCODE2
        ) 
    {
        return BMZ00050_INVALID_FRAME;
    }
    
    uint8_t length = read();
    if (0 != (uint8_t)(length + read())) {   // checksum of length
        return BMZ00050_INVALID_FRAME;
    }
    
    uint8_t cmd = command + 1;               // response command
    if (BMZ00050TOHOST != read() || (cmd) != read()) {
        return BMZ00050_INVALID_FRAME;
    }
    
    length -= 2;
    if (length > len)
    {
        return BMZ00050_NO_SPACE;  // not enough space
    }
    
    DMSG("read:  ");
    DMSG_HEX(cmd);
    
    uint8_t sum = BMZ00050TOHOST + cmd;
    for (uint8_t i = 0; i < length; i++) 
    {
        buf[i] = read();
        sum += buf[i];
        
        DMSG_HEX(buf[i]);
    }
    DMSG('\n');
    
    uint8_t checksum = read();
    if (0 != (uint8_t)(sum + checksum)) {
        DMSG("checksum is not ok\n");
        return BMZ00050_INVALID_FRAME;
    }
    read();         // POSTAMBLE
    
    return length;
}

int8_t BMZ00050_I2C::readAckFrame()
{
    const uint8_t BMZ00050_ACK[] = {0, 0, 0xFF, 0, 0xFF, 0};
    uint8_t ackBuf[sizeof(BMZ00050_ACK)+5];

    DMSG("wait for ack at : ");
    DMSG(millis());
    DMSG('\n');    
    uint16_t time = 0;
    do {
        if (_wire->requestFrom(BMZ00050_I2C_ADDRESS,  sizeof(BMZ00050_ACK) + 1))
        {
            if (read() & 1)
            {                   // check first byte --- IICSR 
                break;         // BMZ00050 is ready
            }
        }

        delay(1);
        time++;
        if (time > BMZ00050_ACK_WAIT_TIME) 
        {
            DMSG("Time out when waiting for ACK\n");
            return BMZ00050_TIMEOUT;
        }
    } while (1); 
    
    DMSG("ready at : ");
    DMSG(millis());
    DMSG('\n');
    
 
    for (uint8_t i = 0; i < sizeof(BMZ00050_ACK); i++)    //Read ack
    {
        ackBuf[i] = read();
        delayMicroseconds(20);
    }  

    if (memcmp(ackBuf, BMZ00050_ACK, sizeof(BMZ00050_ACK))) {
        DMSG("Invalid ACK\n");
        return BMZ00050_INVALID_ACK;
    }
    
    return 0;
}
/************************************************************/
int8_t BMZ00050_I2C:: T_writeCommand(const uint8_t *header, uint8_t hlen, const uint8_t *body = 0, uint8_t blen = 0)
{
    _wire->beginTransmission(BMZ00050_I2C_ADDRESS);
    
    for (uint8_t i = 0; i < hlen; i++)
    {         
        write(header[i]);
    }
    for (uint8_t i = 0; i < hlen; i++)
    {         
        write(header[i]);
    }    
    _wire->endTransmission();

    return 0;
}
int16_t BMZ00050_I2C::T_readResponse(uint8_t buf[], uint8_t len, uint16_t timeout)
{
    uint8_t tmp[2];
    uint8_t time_count=0;
    DMSG("\nRead:  ");    
    uint8_t length[1];  
    do
   {
      time_count ++;
      delay(1);
      if(time_count > 20)
      {
         return BMZ00050_TIMEOUT;
      }   
    
    if(digitalRead(INT) == 0)  
    {
      
        if (_wire->requestFrom(BMZ00050_I2C_ADDRESS, len))     
        {
          tmp[0] = read();   //  header
          DMSG_HEX(tmp[0]);
          tmp[1] = read();   //command;       
          DMSG_HEX(tmp[1]);
  
          length[0] = read();
          DMSG_HEX(length[0]);
          for (uint8_t i = 0; i < length[0]; i++) 
          {
              buf[i] = read();
          }
         }
       }
       break;
   }while(1);
   return length[0];
}
uint8_t BMZ00050_I2C::Rx_Data_Packet(uint8_t buf[], uint8_t len, uint16_t timeout)
{  
    if(digitalRead(INT) == 0)  
    {   
       if (_wire->requestFrom(BMZ00050_I2C_ADDRESS, 64))        
       {
          buf[0] = read();  //header
          buf[1] = read();  //length
          for (uint8_t i = 0; i < buf[1]; i++)
          {
              buf[i+2] = read();      
          }         
       } 
            return 0;   
    }
     return 1;
}
