/*****************************************************************
File:          BM22D1221_1421.cpp
Author:        Liu, BESTMODULES
Description:   Stepper motor library and DC motor library for Arduino 
History:         
V1.0.1  -- initial version;2023-01-09;Arduino IDE : v1.8.15
**************************************************************************/
#include "BM22D1221_1421.h"
/**********************************************************
Description: DC motor constructor 
Parameters:                                                
Return:             
Others: 0 and 1 connect M1A and M1B, 2 and 3 connect M2A and M2B 
        4 and 5 connect M3A and M3B, 6 and 8 connect M4A and M4B 
***************************************************************************/
BM22D1221_1421::BM22D1221_1421()
{
    _pin1 = 0;
    _pin2 = 1;
    _pin3 = 2;
    _pin4 = 3;
    _pin5 = 4;
    _pin6 = 5;
    _pin7 = 6;
    _pin8 = 8;
 }
 /****************************************************************************************
Description: Stepper motor constructor 
Parameters:  interface: driving mode:
                                     FULL4WIRE/1: fall step drive,(200 steps/circle) 
                                     HALF4WIRE/2: half step drive,(400 steps/circle)
             pin1, pin2, pin3 and pin4: Stepper motor connect ports,and respectively connected to ports 0, 1, 2 and 3 
Return:             
Others:      The wiring port can be changed as required
*****************************************************************************************/
BM22D1221_1421::BM22D1221_1421(uint8_t interface, uint8_t pin1, uint8_t pin2, uint8_t pin3, uint8_t pin4)
{ 
    _pin1 = pin1;
    _pin2 = pin2;
    _pin3 = pin3;
    _pin4 = pin4;
    _interface = interface;
    _lastStepTime = 0;
    _stepInterval = 0;
    _currentPos = 0;
    _targetPos = 0;
    _speed = 0;
    _maxSpeed = 1;
    _acceleration = 0;
    _count = 0;
    _initStepInterval = 0;
    _nextStepInterval = 0;
    _minStepInterval = 1;
    _flagAccelerate = 0;
    _flagHoldTorque = 0;
    _everyInitPosition = 0;
    _flagStop = 1;
}
/**********************************************************
Description: BM22D1221_1421 initial 
Parameters:                   
Return:             
Others: Pin set to output mode and low level   
**********************************************************/
void BM22D1221_1421::begin()
{
    pinMode(_pin1, OUTPUT);
    pinMode(_pin2, OUTPUT);
    pinMode(_pin3, OUTPUT);
    pinMode(_pin4, OUTPUT);

    pinMode(_pin5, OUTPUT);
    pinMode(_pin6, OUTPUT);
    pinMode(_pin7, OUTPUT);
    pinMode(_pin8, OUTPUT);


    digitalWrite(_pin1, LOW);
    digitalWrite(_pin2, LOW);
    digitalWrite(_pin3, LOW);
    digitalWrite(_pin4, LOW);

    digitalWrite(_pin5, LOW);
    digitalWrite(_pin6, LOW);
    digitalWrite(_pin7, LOW);
    digitalWrite(_pin8, LOW);

}
/******************************************************************************
Description: Rotating DC motor  
Parameters:  ch: DC motor number:
                                  1: DC motor 1
                                  2: DC motor 2
                                  3: DC motor 3
                                  4: DC motor 4                                  
             rank: Speed gear,range -100~100,positive is clockwise, 
                   negative is counterclockwise, absolute value indicates gear     
Return:             
Others:      
******************************************************************************/
void BM22D1221_1421::dcMotorRun(uint8_t ch, int8_t rank)
{  
    if (_interface==0)
    {
        rank = constrain(rank, -100, 100);
        int a = abs(rank);
        int _rank = map(a, 0, 100, 0, 255);
        if (ch == DC1)
        {
            if (rank >= 0)      
            {
                pinMode(_pin1, OUTPUT);
                digitalWrite(_pin1, LOW);             
                analogWrite(_pin2, _rank);
            } 
            else
            {
                pinMode(_pin2, OUTPUT);
                digitalWrite(_pin2, LOW);             
                analogWrite(_pin1, _rank);  
            }
            _rank1 = rank;
        }
        else if (ch == DC2)
        {
            if (rank >= 0)      
            {
                pinMode(_pin3, OUTPUT);
                digitalWrite(_pin3, LOW);             
                analogWrite(_pin4, _rank);
            } 
            else
            {
                pinMode(_pin4, OUTPUT);
                digitalWrite(_pin4, LOW);             
                analogWrite(_pin3, _rank);  
            }
            _rank2 = rank;
        }


        if (ch == DC3)
        {
            if (rank >= 0)      
            {
                pinMode(_pin5, OUTPUT);
                digitalWrite(_pin5, LOW);             
                analogWrite(_pin6, _rank);
            } 
            else
            {
                pinMode(_pin6, OUTPUT);
                digitalWrite(_pin6, LOW);             
                analogWrite(_pin5, _rank);  
            }
            _rank3 = rank;
        }
        else if (ch == DC4)
        {
            if (rank >= 0)      
            {
                pinMode(_pin7, OUTPUT);
                digitalWrite(_pin7, LOW);             
                analogWrite(_pin8, _rank);
            } 
            else
            {
                pinMode(_pin8, OUTPUT);
                digitalWrite(_pin8, LOW);             
                analogWrite(_pin7, _rank);  
            }
            _rank4 = rank;
        }



    }
}
/**********************************************************
Description: DC motor stop  
Parameters:  ch: DC motor number:
                                  1: DC motor 1
                                  2: DC motor 2 
                                  3: DC motor 3
                                  4: DC motor 4                                                          
Return:             
Others:      
**********************************************************/
void BM22D1221_1421::dcMotorStop(uint8_t ch)
{
    if (_interface==0)
    {
        if (ch == DC1)
        {
            pinMode(_pin1, OUTPUT);
            pinMode(_pin2, OUTPUT);
            digitalWrite(_pin1, LOW);
            digitalWrite(_pin2, LOW);
        }
        if (ch == DC2)
        {
            pinMode(_pin3, OUTPUT);
            pinMode(_pin4, OUTPUT);
            digitalWrite(_pin4, LOW);
            digitalWrite(_pin3, LOW);
        } 
        if (ch == DC3)
        {
            pinMode(_pin5, OUTPUT);
            pinMode(_pin6, OUTPUT);
            digitalWrite(_pin6, LOW);
            digitalWrite(_pin5, LOW);
        } 
        if (ch == DC4)
        {
            pinMode(_pin7, OUTPUT);
            pinMode(_pin8, OUTPUT);
            digitalWrite(_pin7, LOW);
            digitalWrite(_pin8, LOW);
        } 

    }
}
/**********************************************************
Description: DC motor brake  
Parameters:  ch: DC motor number:
                                  1: DC motor 1
                                  2: DC motor 2 
                                  3: DC motor 3
                                  4: DC motor 4                                                     
Return:             
Others:      
**********************************************************/
void BM22D1221_1421::dcMotorBrake(uint8_t ch)
{
    if (_interface==0)
    {
        if (ch == DC1)
        {
            pinMode(_pin1, OUTPUT);
            pinMode(_pin2, OUTPUT);
            digitalWrite(_pin1, HIGH);
            digitalWrite(_pin2, HIGH);
        }
        if (ch == DC2)
        {
            pinMode(_pin3, OUTPUT);
            pinMode(_pin4, OUTPUT);
            digitalWrite(_pin3, HIGH);
            digitalWrite(_pin4, HIGH);
        }  
        if (ch == DC3)
        {
            pinMode(_pin5, OUTPUT);
            pinMode(_pin6, OUTPUT);
            digitalWrite(_pin5, HIGH);
            digitalWrite(_pin6, HIGH);
        }
        if (ch == DC4)
        {
            pinMode(_pin7, OUTPUT);
            pinMode(_pin8, OUTPUT);
            digitalWrite(_pin7, HIGH);
            digitalWrite(_pin8, HIGH);
        }  

    } 
}
/************************************************************************
Description: Get DC motor speed gear  
Parameters:  ch: DC motor number:
                                  1: DC motor 1
                                  2: DC motor 2 
                                  3: DC motor 3
                                  4: DC motor 4                                                   
Return:      DC motor speed gear: -100~100    
Others:      
************************************************************************/
int8_t BM22D1221_1421::getDcMotorRank(int8_t ch)
{
    if (_interface==0)
    {
        if (ch == DC1)
        {
            return(_rank1);
        }
        if (ch == DC2)
        {
            return(_rank2);
        } 
        if (ch == DC3)
        {
            return(_rank3);
        }
        if (ch == DC4)
        {
            return(_rank4);
        } 


        return(0);
    }
    else 
    {
        return(0);
    }   
}
/**********************************************************
Description: The stepper motor rotates to the absolute position at a constant speed  
Parameters:  absolute: Absolute position,a position relative to the zero position,steps(unit)
             speed: The speed of the stepper motor,steps per second(unit)  
                    It is recommended not to exceed 800 steps/s under full step
                    It is recommended not to exceed 1600 steps/s under half step                    
Return:               
Others:      This function runs the "block" program. That is, Arduino will not
             continue to execute the subsequent program content until the target location is reached.     
**********************************************************/
void BM22D1221_1421::stepperMoveTo(int32_t  absolute, uint16_t speed)
{  
    if (_interface)
    {
        if (_targetPos != absolute)
        {   
            _flagStop = 0;
            _flagAccelerate = 0;  
            _targetPos = absolute;
            _dir = (_targetPos - _currentPos > 0) ? FORWARD : REVERSAL; 
      
        }  
         setSpeed(speed);
         stepperKeepRun(); 
    }
}
/**********************************************************
Description: The stepper motor rotates to the absolute position with acceleration      
Parameters:  absolute: Absolute position,a position relative to the zero position,steps(unit)                     
Return:                
Others:      This function runs the "block" program. That is, Arduino will not
             continue to execute the subsequent program content until the target location is reached.
**********************************************************/
void BM22D1221_1421::stepperMoveTo(int32_t absolute)
{
    if (_interface)
    {
        if (_targetPos != absolute )
        {
            _flagStop = 0;
            _targetPos = absolute;
            _flagAccelerate = 1;
            computeNewSpeed();
            stepperKeepRun();
        }
    }
}
/**********************************************************
Description: How many steps does the stepper motor rotate at a constant speed    
Parameters:  relative: Rotation steps of stepper motor, positive indicates clockwise, 
                       negative indicates counterclockwise,steps(unit)
             speed: The speed of the stepper motor,steps per second(unit)                      
Return:                
Others:      This function runs the "block" program. That is, Arduino will not
             continue to execute the subsequent program content until the target location is reached.
**********************************************************/
void BM22D1221_1421::stepperMove(int32_t relative, uint16_t speed)
{
    if (_interface)
    {
        stepperMoveTo(_currentPos + relative, speed);
    }
}
/**********************************************************
Description: How many steps does the stepping motor rotate with acceleration     
Parameters:  relative:Rotation steps of stepper motor, positive indicates clockwise, 
                      negative indicates counterclockwise,steps(unit)                   
Return:               
Others:      This function runs the "block" program. That is, Arduino will not
             continue to execute the subsequent program content until the target location is reached.
**********************************************************/
void BM22D1221_1421::stepperMove(int32_t  relative)
{
    if (_interface)
    {
        stepperMoveTo(_currentPos + relative);
    }
}
/**********************************************************
Description: Get maximum speed of stepper motor         
Parameters:   
Return:      Maximum speed of stepper motor,steps per second(unit)      
Others:      
**********************************************************/
uint16_t BM22D1221_1421::getStepperMaxSpeed()
{
    if (_interface)
    {
        return _maxSpeed;
    }
    else 
    {
        return(0);
    }    
}
/**********************************************************
Description: Get acceleration of stepper motor       
Parameters:  
Return:      acceleration of stepper motor,step per square second(unit)      
Others:      
**********************************************************/
uint16_t BM22D1221_1421::getStepperAcceleration()
{
    if (_interface)
    {
        return _acceleration;
    }
    else 
    {
        return(0);
    }    
}
/**********************************************************
Description: Get the current position of stepper motor        
Parameters:   
Return:      The current position of stepper motor,steps(unit)     
Others:      
**********************************************************/
int32_t BM22D1221_1421::getStepperPosition()
{
    if (_interface)
    {
        return _currentPos;
    }
    else 
    {
        return(0);
    }    
}
/**********************************************************
Description: Set the maximum speed of the stepper motor  
Parameters:  maxSpeed: The maximum speed of the stepper motor,     
                       steps per second(unit)   
                       It is recommended not to exceed 800 steps/s under full step
                       It is recommended not to exceed 1600 steps/s under half step                
Return:             
Others:            
**********************************************************/
void BM22D1221_1421::setStepperMaxSpeed(uint16_t maxSpeed)
{
    if (_interface)
    {
        if (_maxSpeed != maxSpeed)
        {
           
            _maxSpeed = (float)maxSpeed;
            _minStepInterval = 1000000 / maxSpeed;      
            if (_count > 0)   // Recompute _count from current speed and adjust speed if accelerating or cruising
            {
                _count = (long)((_speed * _speed) / (2 * _acceleration)); 
                computeNewSpeed();
            }
        }
    }
}
/**********************************************************
Description: Set the acceleration of the stepper motor  
Parameters:  acceleration: The acceleration of the stepper motor,step per square second(unit)                       
Return:             
Others:      
**********************************************************/
void BM22D1221_1421::setStepperAcceleration(uint16_t acceleration)
{
    if (_interface)
    {
        if (acceleration == 0)
        {
            return;
        }
        if (_acceleration != acceleration)
        {
            _count = _count * (_acceleration / ((float)acceleration));  // Recompute _count  
            _initStepInterval = 0.676 * sqrt(2 / ((float)acceleration)) * 1000000; // New c0
            _acceleration = (float)acceleration;
        }
    }
}
/**********************************************************
Description: Set the current position of stepping motor           
Parameters:  position: Position of stepper motor,steps(unit)                     
Return:                
Others:      
**********************************************************/
void BM22D1221_1421::setStepperCurrentPosition(int32_t position)
{
    if (_interface)
    {
        _targetPos = _currentPos = position;
        _count = 0;
        _stepInterval = 0;
        _speed = 0;
        digitalWrite(_pin1, LOW);
        digitalWrite(_pin2, LOW);
        digitalWrite(_pin3, LOW);
        digitalWrite(_pin4, LOW);  
    }      
}
/**********************************************************
Description: Set the speed of the stepper motor  
Parameters:  speed:The speed of the stepper motor,steps per second(unit)                             
Return:             
Others:      It requires less than the maximum speed
**********************************************************/
void BM22D1221_1421::setSpeed(uint16_t speed)
{
    if (_interface)
    {
        if (speed == _speed)
        {
            return;
        }
        speed = constrain(speed, -_maxSpeed, _maxSpeed);
        if (speed == 0)
        {
            _stepInterval = 0;
        }
        else
        {
            _stepInterval = fabs(1000000 / speed);
        }
        _speed = fabs(speed);
        _speed=(float)_speed;
    }
}
/**********************************************************
Description: Calculate the new instantaneous speed and set it to current speed       
Parameters:                                  
Return:                
Others:      
**********************************************************/
void BM22D1221_1421::computeNewSpeed()
{
    if (_interface)
    {
        _distanceTo = getStepperDistanceToGo(); 
        long stepsToStop = (long)((_speed * _speed) / (2 * _acceleration)); 
        if (_distanceTo == 0 && stepsToStop <= 1)
        {
            _nextStepInterval = _nextStepInterval - ((2 * _nextStepInterval) / ((4 * _count) + 1)); 
            _nextStepInterval = max(_nextStepInterval, _minStepInterval); 
            _count++;
            _stepInterval = _nextStepInterval;
            _speed = 1000000 / _nextStepInterval;
            if (_dir == REVERSAL)
           {
               _speed = -_speed;
           }  
            return;
        }
        if (_distanceTo > 0) // Need to go clockwise from here, maybe decelerate now
        {      
            if (_count > 0) // Currently accelerating
            {  
                if ((stepsToStop >= _distanceTo) || _dir == REVERSAL) //Need to decel now?
                _count = -stepsToStop; // Start deceleration
            }
            else if (_count < 0)       // Currently decelerating
            {           
                if ((stepsToStop < _distanceTo) && _dir == FORWARD)   //Need to accel again?
                _count = -_count; // Start accceleration
            }
        }
        else if (_distanceTo < 0) // Need to go anticlockwise from here, maybe decelerate
        {       
            if (_count > 0) // Currently accelerating
            {
                if ((stepsToStop >= -_distanceTo) || _dir == FORWARD) //Need to decel now?
                _count = -stepsToStop; // Start deceleration
            }
            else if (_count < 0)       // Currently decelerating
            {
                if ((stepsToStop < -_distanceTo) && _dir == REVERSAL) //Need to accel again?
                _count = -_count; // Start accceleration
            }
        }  
        if (_count == 0) // Need to accelerate or decelerate
        {    
            _nextStepInterval = _initStepInterval;   // First step from stopped
            _dir = (_distanceTo > 0) ? FORWARD : REVERSAL;
        }
        else    // Subsequent step. Works for accel and decel
        {
            _nextStepInterval = _nextStepInterval - ((2 * _nextStepInterval) / ((4 * _count) + 1)); 
            _nextStepInterval = max(_nextStepInterval, _minStepInterval); 
        }
        _count++;
        _stepInterval = _nextStepInterval;
        _speed = 1000000 / _nextStepInterval;
        if (_dir == REVERSAL)
        {
            _speed = -_speed;
        }
    }
}
/**********************************************************
Description: Stepping function and half stepping function of stepper motor         
Parameters:  step: Running steps of stepper motor,steps(unit)                    
Return:               
Others:      
**********************************************************/
void BM22D1221_1421::step (long step)
{
    if (_interface == FULL4WIRE)//Full step drive
    {
        switch (step & 0x3)
        {
            case 0://1011
                digitalWrite(_pin1, HIGH);    
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, HIGH);
                digitalWrite(_pin4, HIGH);
                break;
            case 1://1110
                digitalWrite(_pin1, HIGH);    
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, HIGH);
                digitalWrite(_pin4, LOW);
                break;
            case 2://0111
                digitalWrite(_pin1, LOW);    
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, HIGH);
                digitalWrite(_pin4, HIGH);
                break;
            case 3://1101
                digitalWrite(_pin1, HIGH);    
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, LOW);
                digitalWrite(_pin4, HIGH);
                break;
        }
    }
    else//Half step drive  
    {       
        switch (step & 0x7)
        {
            case 0://1000
                digitalWrite(_pin1, HIGH);    
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, LOW);
                digitalWrite(_pin4, LOW);
                break;
            case 1://1010
                digitalWrite(_pin1, HIGH);
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, HIGH);                
                digitalWrite(_pin4, LOW);
                break;
            case 2://0010
                digitalWrite(_pin1, LOW);
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, HIGH);        
                digitalWrite(_pin4, LOW);
                break;
            case 3://0110
                digitalWrite(_pin1, LOW);
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, HIGH);        
                digitalWrite(_pin4, LOW); 
                break;
            case 4://0100
                digitalWrite(_pin1, LOW);
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, LOW);                
                digitalWrite(_pin4, LOW);
                break;
            case 5://0101
                digitalWrite(_pin1, LOW);
                digitalWrite(_pin2, HIGH);
                digitalWrite(_pin3, LOW);                
                digitalWrite(_pin4, HIGH);
                break;
            case 6://0001
                digitalWrite(_pin1, LOW);
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, LOW);                
                digitalWrite(_pin4, HIGH);
                break;
            case 7://1001
                digitalWrite(_pin1, HIGH);
                digitalWrite(_pin2, LOW);
                digitalWrite(_pin3, LOW);                
                digitalWrite(_pin4, HIGH);
                break;          
        }
    }
}
/**********************************************************
Description: Stepper motor runs one step   
Parameters:                           
Return:      true:Success
             false:Failed       
Others:      
**********************************************************/
boolean BM22D1221_1421::run()
{   
    if (_interface)
    { 
        if (!_stepInterval)  // Dont do anything unless we actually have a step interval
        {
            return false;
        }
        unsigned long time = micros();   
        if (time - _lastStepTime >= _stepInterval)
        {
            if (_dir == FORWARD)
            {
                _currentPos += 1;  // FORWARD
            }
            else
            {
                _currentPos -= 1;  // REVERSAL  
            } 
            step(_currentPos);    
            _lastStepTime = time; // Caution: does not account for costs in step()
            return true;
        }
        else
        {
            return false;
        }
    }
    else 
    {
        return(0);
    }    
} 
/**********************************************************
Description: Stepper motor rotation     
Parameters:                       
Return:           
Others:     
**********************************************************/
void BM22D1221_1421::stepperKeepRun()
{   
    if (_interface)
    {
        _distanceTo = getStepperDistanceToGo();
        while (_flagStop == 0 && _flagAccelerate == 0) //The stepper motor rotates at a constant speed
        {
            if (run())
            {
                _distanceTo = getStepperDistanceToGo();
            } 
            else
            {
                if (getStepperPosition() != _everyInitPosition)
                {
                    if (_interface == HALF4WIRE)
                    {
                        step(_currentPos);
                        if (fabs(_speed) < 800)
                        {
                            delayMicroseconds(25); 
                            digitalWrite(_pin1, HIGH);
                            digitalWrite(_pin2, HIGH);
                            digitalWrite(_pin3, HIGH);
                            digitalWrite(_pin4, HIGH);
                        }
                        else
                        {
                            delayMicroseconds(80);
                            digitalWrite(_pin1, HIGH);
                            digitalWrite(_pin2, HIGH);
                            digitalWrite(_pin3, HIGH);
                            digitalWrite(_pin4, HIGH);
                        }
                    }            
                }
            }
            if (_distanceTo == 0)     
            {
                while (1)
                {                    
                    unsigned long time1 = micros();   
                    if ((time1 - _lastStepTime >= _stepInterval) && (time1 - _lastStepTime < 5*_stepInterval))  //Maintain five steps to release current
                    {        
                        digitalWrite(_pin1, HIGH);
                        digitalWrite(_pin2, HIGH);
                        digitalWrite(_pin3, HIGH);
                        digitalWrite(_pin4, HIGH);
                    }
                    else if (time1 - _lastStepTime >= 5*_stepInterval)
                    {
                        _lastStepTime = time1;
                        digitalWrite(_pin1, LOW);
                        digitalWrite(_pin2, LOW);
                        digitalWrite(_pin3, LOW);
                        digitalWrite(_pin4, LOW);
                        _flagStop = 1;
                        _speed = 0;
                        _stepInterval = 0 ;
                        _everyInitPosition = getStepperPosition();         
                        break; 
                    }
                    else
                    {
                        if (_interface == HALF4WIRE)
                        {
                            step(_currentPos);
                            if (fabs(_speed) < 800)
                            {
                                delayMicroseconds(25); 
                                digitalWrite(_pin1, HIGH);
                                digitalWrite(_pin2, HIGH);
                                digitalWrite(_pin3, HIGH);
                                digitalWrite(_pin4, HIGH);
                            }
                            else
                            {
                                delayMicroseconds(80);
                                digitalWrite(_pin1, HIGH);
                                digitalWrite(_pin2, HIGH);
                                digitalWrite(_pin3, HIGH);
                                digitalWrite(_pin4, HIGH);
                            }  
                        }
                    }
                }
                break;
            }
        }     
        while (_flagStop == 0 && _flagAccelerate == 1) //The stepper motor rotates at an acceleration
        {      
            if (run())
            {
                computeNewSpeed();
            }
            else  
            {
                if (getStepperPosition() != _everyInitPosition)
                {
                    if (_interface == HALF4WIRE)
                    {
                        step(_currentPos);
                        if (fabs(_speed) < 800)
                        {
                            delayMicroseconds(25); 
                        }
                        else
                        {
                            delayMicroseconds(80);
                        }
                        digitalWrite(_pin1, HIGH);
                        digitalWrite(_pin2, HIGH);
                        digitalWrite(_pin3, HIGH);
                        digitalWrite(_pin4, HIGH);
                    }           
                }
            }
            if (_distanceTo == 0)
            {     
                while (1)
                {                 
                    unsigned long time2 = micros();   
                    if ((time2 - _lastStepTime >= _stepInterval) && (time2 - _lastStepTime < 5*_stepInterval)) //Maintain five steps to release current
                    {
                        digitalWrite(_pin1, HIGH);
                        digitalWrite(_pin2, HIGH);
                        digitalWrite(_pin3, HIGH);
                        digitalWrite(_pin4, HIGH);
                    }
                    else if (time2 - _lastStepTime >= 5*_stepInterval)
                    {
                        _lastStepTime = time2;
                        digitalWrite(_pin1, LOW);
                        digitalWrite(_pin2, LOW);
                        digitalWrite(_pin3, LOW);
                        digitalWrite(_pin4, LOW);
                        _flagStop = 1;
                        _flagAccelerate = 0;
                        _speed = 0;
                        _stepInterval = 0 ;
                        _count = 0;
                        _everyInitPosition = getStepperPosition();         
                        break; 
                    }
                    else
                    {
                        if (_interface == HALF4WIRE)
                        {
                            step(_currentPos);
                            if (fabs(_speed) < 800)
                            {
                                delayMicroseconds(20); 
                            }
                            else
                            {
                                delayMicroseconds(80);
                            }
                            digitalWrite(_pin1, HIGH);
                            digitalWrite(_pin2, HIGH);
                            digitalWrite(_pin3, HIGH);
                            digitalWrite(_pin4, HIGH);
                        }   
                    }
                }
                break;
            }
        }
    }
}
/**********************************************************
Description: Get the distance between the current position of the stepper motor and the target position        
Parameters:   
Return:      The distance between the current position of the stepper motor and the target position,steps(unit)      
Others:      
**********************************************************/
int32_t BM22D1221_1421::getStepperDistanceToGo()
{
    if (_interface)
    {
        return _targetPos - _currentPos;
    }
    else 
    {
        return(0);
    }    
}
