/**
 * @file AP_BIA8_IQMode.c
 * @author 陳挺 (tony@e-besthealth.com)
 * @brief IQ Mode阻抗掃描，尋找相關性
 * @version 0.1
 * @date 2019-12-04
 *
 * @copyright BEST HEALTH ELECTRONIC INC (c) 2019
 *
 * Compiler Ver. : V3.57
 * IDE3000  Ver. : V8.02
 * MCU / CFG Ver. :BH66F2663/1.3
 */
#include "..\BH66F2663_Drv\Project_Global.h"
#include "BH66F2663.h"
#include "..\BH66F2663_Drv\Drv_AFE_ADC.h"
#include "..\BH66F2663_Drv\Drv_AFE_Power.h"
#include "..\BH66F2663_Drv\Drv_BIA.h"

#include "..\BH66F2663_Lib\SDK_ADCFilter_IIR.h"
#include "AP_BIA8_IQ.h"
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@阻抗變設定@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
volatile typedefImpedance *IQPointerImpedanceIQ;

volatile typedefImpedance BIA8Channel_ADC_Re;
volatile typedefImpedance BIA8Channel_ADC_IM;

BIA8IQScan_t BIA8IQScan;

#define Impedance_RF1   14.96f        // 實際參考電阻阻值放大100倍，例RF1 = 14.96ohm
#define Impedance_RF2   98.39f		// 98.39f
#define Impedance_RF3   839.10f		// 839.10f
#define Impedance_RF4   914.96f		// 914.96f

#include <math.h>
#define M_PI_1  3.14159265358979
#define M_PI_2  1.57079632679489

/**
 * @brief 根據實部和虛部值計算相角值
 *
 * @param re 實部值,MOD1ON~MOD1OP
 * @param im 虛部值,MOD2ON~MOD2OP
 * @return double v 相角值
 */
double GetPhase(double re,double im)
{
    double v;
    if(fabs(re) >= fabs(im))
    {
        v = atan(im/re);
        if( re < 0.0)
        {
            if(im >= 0.0)
                v += M_PI_1;
            else
                v -= M_PI_1;
        }
    }
    else
    {
        v = -atan(re/im);
        if(im < 0.)
            v -= M_PI_2;
        else
            v += M_PI_2;
    }
    v = v*180 / M_PI_1;
    return v;
}

/**
 * @brief 根據實部和虛部值計算Z值
 *
 * @param a 實部值,MOD1ON~MOD1OP
 * @param b 虛部值,MOD2ON~MOD2OP
 * @return c Z_adc值
 */
double Pythagoreantheorem(double re,double im)
{
    double c;
    c = (double)re*re+(double)im*im;
    c = sqrt(c);
    return c;
}

void GetBIA8IQ_Phase()
{
	volatile float PhaseRef;
	PhaseRef = (BIA8IQScan.fitterADC_RF2_Phase + BIA8IQScan.fitterADC_RF3_Phase + BIA8IQScan.fitterADC_RF4_Phase)/3;
	IQPointerImpedanceIQ = &BIA8IQScan.TwoLegs_Phase;
	unsigned char i;
	for ( i = 0; i < 11; i++)
	{
		if (*IQPointerImpedanceIQ == 0)
		{
			*IQPointerImpedanceIQ = 0x00;
		}
		else
		{
			*IQPointerImpedanceIQ = PhaseRef - *IQPointerImpedanceIQ;
		}
		IQPointerImpedanceIQ++;
		IQPointerImpedanceIQ++;
	}

}

/**
 * @brief 根據ADC值計算阻抗值
 *
 */
void GetBIA8IQ_Z()
{
	IQPointerImpedanceIQ = &BIA8IQScan.TwoLegs_Z;
	unsigned char i;
	for (i = 0; i < 11; i++)
	{
		if (*IQPointerImpedanceIQ == 0)
		{
			*IQPointerImpedanceIQ = 0x00;
		}
		else if (*IQPointerImpedanceIQ < BIA8IQScan.fitterADC_RF2_Z)
		{
			*IQPointerImpedanceIQ = (Impedance_RF1 + ((*IQPointerImpedanceIQ-BIA8IQScan.fitterADC_RF1_Z)*(Impedance_RF2 - Impedance_RF1))/(BIA8IQScan.fitterADC_RF2_Z-BIA8IQScan.fitterADC_RF1_Z));
		}
		else if (*IQPointerImpedanceIQ < BIA8IQScan.fitterADC_RF3_Z)
		{
			*IQPointerImpedanceIQ = (Impedance_RF2 + ((*IQPointerImpedanceIQ-BIA8IQScan.fitterADC_RF2_Z)*(Impedance_RF3 - Impedance_RF2))/(BIA8IQScan.fitterADC_RF3_Z-BIA8IQScan.fitterADC_RF2_Z));
		}
		else
		{
			*IQPointerImpedanceIQ = (Impedance_RF3 + ((*IQPointerImpedanceIQ-BIA8IQScan.fitterADC_RF3_Z)*(Impedance_RF4 - Impedance_RF3))/(BIA8IQScan.fitterADC_RF4_Z-BIA8IQScan.fitterADC_RF3_Z));
		}
		IQPointerImpedanceIQ++;
		IQPointerImpedanceIQ++;
	}
}

/**
 * @brief 八電極阻抗相角測量
 * @note RF1=R7//(R9+R10+R11) RF2=R9//(R7+R10+R11) RF3=R10//(R7+R7+R11) RF4=R11//(R7+R9+R10)
 * @note 開發板V1.2 R7=15ohm  R9=100ohm R10=1000ohm R11=5100ohm
 * @note 開發板V1.4 R7=15ohm R10=100ohm R11=1000ohm R9=5100ohm  開發板V1.4參考電阻的計算方式: RF1=R7//(R9+R10+R11) RF2=R10//(R7+R7+R11) RF3=R11//(R7+R9+R10) RF4=R9//(R7+R10+R11)
 *
 */
void AP_BIA8_IQ_Loop()
{
	static unsigned char ChannelState;
	if (BIA8IQScan.statusSet == MODE_INIT)
	{
		// AFE Power
		Drv_VOREG_OUTPUT(VOREG_POWERON_2V6);
		// SINE WAVE
		Drv_BIAPowerOn();

		ChannelState = MODE_INIT;
		BIA8IQScan.statusRead = BIA8_CHANNEL_RF1;
		IQPointerImpedanceIQ = &BIA8IQScan.fitterADC_RF1_Z;
		// ADC
		SET_VREF_GAIN_0_5();
		SET_ADC_GAIN_1();
		SET_PGA_GAIN_1();
		SET_ADC_DATARATE3_64HZ();
		SDKADCFilterData.StableThreshold = 50;
		SDKADCFilterData.StableCntTag  = 10;
		BIA8IQScan.statusSet = MODE_STATE;
	}
	else if (BIA8IQScan.statusSet == MODE_STATE)
	{
		switch (ChannelState)
		{
		case MODE_INIT:
			if (BIA8Setting.Channel & BIA8IQScan.statusRead)
			{
				SET_ADC_CHANNEL_MODE_IQ_R();
				Drv_BIA8ChannelSwich(BIA8Setting.Channel & BIA8IQScan.statusRead);
				Drv_ADCStart();
				// 濾波設置初始化
				fun_FilterInit();
				ChannelState = MODE_STATE;
			}
			else
			{
				if (BIA8IQScan.statusRead == BIA8_CHANNEL_TRUNK)
				{
					*IQPointerImpedanceIQ = 0;
					IQPointerImpedanceIQ++;
					*IQPointerImpedanceIQ = 0;
					IQPointerImpedanceIQ++;
					GetBIA8IQ_Phase();
					GetBIA8IQ_Z();
					BIA8IQScan.statusRead = BIA8_FINISH;
				}
				else
				{
					*IQPointerImpedanceIQ = 0;
					IQPointerImpedanceIQ++;
					*IQPointerImpedanceIQ = 0;
					IQPointerImpedanceIQ++;
					BIA8IQScan.statusRead <<= 1;
				}
			}
			break;
		case MODE_STATE:
			fun_Filtering();
			if (SDKADCFilterData.flag.b.IsStable)
			{
				if (_pgacs == 0x43)
				{
					BIA8Channel_ADC_Re = SDKADCFilterData.Current;
					SET_ADC_CHANNEL_MODE_IQ_I();
					Drv_ADCStart();
					// 濾波設置初始化
					fun_FilterInit();
				}
				else
				{
					BIA8Channel_ADC_IM = SDKADCFilterData.Current;

					*IQPointerImpedanceIQ = Pythagoreantheorem(BIA8Channel_ADC_IM, BIA8Channel_ADC_Re);
					IQPointerImpedanceIQ++;
					*IQPointerImpedanceIQ = GetPhase(BIA8Channel_ADC_IM, BIA8Channel_ADC_Re);
					IQPointerImpedanceIQ++;

					if (BIA8IQScan.statusRead == BIA8_CHANNEL_TRUNK)
					{
						GetBIA8IQ_Phase();
						GetBIA8IQ_Z();
						BIA8IQScan.statusRead = BIA8_FINISH;
					}
					else
					{
						BIA8IQScan.statusRead <<= 1;
					}
					ChannelState = MODE_INIT;
				}
			}
		case MODE_FINISH:
			break;
		default:
			break;
		}
	}
	else if (BIA8IQScan.statusSet == MODE_EXIT)
	{
        Drv_BIAPowerDown();
	}
}
