/*********************************************************************************************************//**
 * @file    BM32S3021_1_HT32/src/BM32S3021_1.c
 * @version V1.0.1
 * @date    2024-08-22
 * @brief   The function of BM32S3021 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM32S3021_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM32S3021_1_SERIAL = BM32S3021_1_SERIAL;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief  module serial number select.
 * @param  serial_number: select serial number.
 * @retval BM32S3021_1_selStatus
                  @arg BM32S3021_1_SUCCESS
                  @arg BM32S3021_1_FAILURE 
 ************************************************************************************************************/
BM32S3021_1_selStatus BM32S3021_1_selSerial(uint32_t serial_number)
{
  if(CHECK_SERIALn(serial_number) == -1)
  {
    return BM32S3021_1_FAILURE;
  }

  gBM32S3021_1_SERIAL = serial_number;
  return BM32S3021_1_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using UART communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM32S3021_1_Init(void)
{
	{
	USART_InitTypeDef USART_InitStructure = {0};
  USART_InitStructure.USART_BaudRate = 9600;
  USART_InitStructure.USART_WordLength = USART_WORDLENGTH_8B;
  USART_InitStructure.USART_StopBits = USART_STOPBITS_1;
  USART_InitStructure.USART_Parity = USART_PARITY_NO;
  USART_InitStructure.USART_Mode = USART_MODE_NORMAL;
  UARTM_Init(gBM32S3021_1_SERIAL, &USART_InitStructure, BM32S3021_1_UART_TXTIMEOUT);
  }
	//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BM32S3021_1_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BM32S3021_1_INTPIN_GPIO_ID, BM32S3021_1_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BM32S3021_1_INTPIN_GPIO_PORT, BM32S3021_1_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BM32S3021_1_INTPIN_GPIO_PORT, BM32S3021_1_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BM32S3021_1_INTPIN_GPIO_PORT, BM32S3021_1_INTPIN_GPIO_PIN, ENABLE);
}

/*********************************************************************************************************//**
 * @brief  Get INT Status
 * @param  void
 * @retval INT Status:
                    @arg 0:INT output low level 
                    @arg 1:INT output high level 
 ************************************************************************************************************/
uint8_t BM32S3021_1_getINT(void)
{
	return (GPIO_ReadInBit(BM32S3021_1_INTPIN_GPIO_PORT, BM32S3021_1_INTPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  Get IR Induction state
 * @param  void
 * @retval BM32S3021_1_irStatus: 
            @arg BM32S3021_1_irStatus&0x08 == 1: calibration
            @arg BM32S3021_1_irStatus&0x08 == 0:
                                BM32S3021_1_irStatus&0x04== : 
                                                 1 : Swipe left 
                                                 0 : not Swipe left 
                                BM32S3021_1_irStatus&0x02== : 
                                                 1 : Swipe right
                                                 0 : not Swipe right
                                BM32S3021_1_irStatus&0x01== : 
                                                 1 : approach   
                                                 0 : not approach    
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRStatus(void)
{
  uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x02, 0x01,0xD8};
  uint8_t BM32S3021_1_irStatus = 0;
  uint8_t BM32S3021_1_buff[6] = {0};
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
  {
   BM32S3021_1_irStatus = BM32S3021_1_buff[4];
  }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_irStatus;
}

/*********************************************************************************************************//**
 * @brief  Enter distance learning mode
 * @param  void
 * @retval Distance learning situation: 
            @arg 0(BM32S3021_1_R_SUCCESS):distance Learn Success   
            @arg 1(BM32S3021_1_R_FAIL):distance Learn Fail
 * @Others Place the object to be measured to the distance you want to learn, and wait for 2s to complete the learning
 ************************************************************************************************************/
uint8_t BM32S3021_1_distanceLearning(void)
{
    uint8_t BM32S3021_1_sendBuf[3] = {0x55, 0x19, 0x6E};
    uint8_t BM32S3021_1_buff[3] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,3); 
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
    {
      if(BM32S3021_1_buff[1]== 0x7F)
      {
        _BM32S3021_1_delay(2500);
         return BM32S3021_1_R_SUCCESS ;
       }
    }
    _BM32S3021_1_delay(2500);
    return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Get number of left & right sliding
 * @param  void
 * @retval BM32S3021_1_number: 
            @arg BM32S3021_1_number>0: number of right sliding
            @arg BM32S3021_1_number<0:number of left sliding
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRGestureNum(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x03, 0x01, 0xD9};
    uint8_t BM32S3021_1_buff[6] = {0};
    uint8_t BM32S3021_1_number = 0;
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_number= BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_number;
}

/*********************************************************************************************************//**
 * @brief  Get the version information
 * @param  void
 * @retval BM32S3021_1_ver:FWver
 * @Others EX.   //High byte     LOW byte
                 //0b00000 0001  0000 0000(version 1.0)
 ************************************************************************************************************/
int16_t BM32S3021_1_getFWVer(void)
{
    uint8_t BM32S3021_1_sendBuf1[5] = {0x55, 0x80, 0x00, 0x01, 0xD6};
    uint8_t BM32S3021_1_sendBuf2[5] = {0x55, 0x80, 0x01, 0x01, 0xD7};
    uint8_t BM32S3021_1_buff[6] = {0};
    int16_t BM32S3021_1_ver;
    uint8_t BM32S3021_1_ver_l = 0;
    uint8_t BM32S3021_1_ver_h = 0;
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf1,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_ver_l = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf2,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_ver_h = BM32S3021_1_buff[4];
    }
    BM32S3021_1_ver = (BM32S3021_1_ver_h<<8)+BM32S3021_1_ver_l ;
    _BM32S3021_1_delay(20);
    return BM32S3021_1_ver;
}

/*********************************************************************************************************//**
 * @brief  Module reset
 * @param  void
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 ************************************************************************************************************/
uint8_t BM32S3021_1_reset(void)
{
    uint8_t BM32S3021_1_sendBuf[3] = {0x55, 0x10, 0x65};
    uint8_t BM32S3021_1_buff[3] = {0};
     _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,3); 
    if( _BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
    {
     if(BM32S3021_1_buff[1]== 0x7f)
     {
      _BM32S3021_1_delay(20);
       return BM32S3021_1_R_SUCCESS;
     }
    }
      _BM32S3021_1_delay(20);
    return BM32S3021_1_R_FAIL ;
}  

/*********************************************************************************************************//**
 * @brief  Get IR debounce value
 * @param  void
 * @retval BM32S3021_1_debounce: Number of debounces(0~255)
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRDebounce(void)
{
  uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x06, 0x01, 0xDC};
  uint8_t BM32S3021_1_debounce = 0;
  uint8_t BM32S3021_1_buff[6] = {0};
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
   {
    BM32S3021_1_debounce = BM32S3021_1_buff[4];
   }
   _BM32S3021_1_delay(20);
  return BM32S3021_1_debounce;
}

/*********************************************************************************************************//**
 * @brief  Get IR trigger threshold
 * @param  void
 * @retval BM32S3021_1_threshold: Trigger threshold(10~200)
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRThreshold(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x07, 0x01, 0xDD};
    uint8_t BM32S3021_1_threshold = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_threshold = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_threshold;
}

/*********************************************************************************************************//**
 * @brief  Get IRQ triger time
 * @param  void
 * @retval BM32S3021_1_IRQTrigerTime: IRQ Triger Time argument(0~255)
 * @Others IRQ Triger Time =  BM32S3021_1_IRQTrigerTime * 4ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRQTrigerTime(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x08, 0x01, 0xDE};
    uint8_t BM32S3021_1_IRQTrigerTime = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_IRQTrigerTime = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_IRQTrigerTime;
}

/*********************************************************************************************************//**
 * @brief  Get IR Cumulative continuous sliding time
 * @param  void
 * @retval BM32S3021_1_IRContinutyGestureTime: Cumulative continuous sliding time argument(0~127)
 * @Others Cumulative continuous sliding time =  BM32S3021_1_IRContinutyGestureTime * 64ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRContinutyGestureTime(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x09, 0x01, 0xDF};
    uint8_t BM32S3021_1_IRContinutyGestureTime = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_IRContinutyGestureTime = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_IRContinutyGestureTime;
}

/*********************************************************************************************************//**
 * @brief  Get IR fastest gesture time
 * @param  void
 * @retval BM32S3021_1_IRFastestGestureTime: fastest gesture time argument(0~200)
 * @Others fastest gesture time =  20 + BM32S3021_1_IRFastestGestureTime * 4ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRFastestGestureTime(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x0A, 0x01, 0xE0};
    uint8_t BM32S3021_1_IRFastestGestureTime = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
	 _BM32S3021_1_delay(100);
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_IRFastestGestureTime = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_IRFastestGestureTime;
}

/*********************************************************************************************************//**
 * @brief  Get IR  slowest  gesture time
 * @param  void
 * @retval BM32S3021_1_IRSlowestGestureTime: slowest  gesture time argument(0~200)
 * @Others slowest  gesture time = BM32S3021_1_IRSlowestGestureTime * 16ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_getIRSlowestGestureTime(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x0B, 0x01, 0xE1};
    uint8_t BM32S3021_1_IRSlowestGestureTime = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
		_BM32S3021_1_delay(100);
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_IRSlowestGestureTime = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_IRSlowestGestureTime;
}

/*********************************************************************************************************//**
 * @brief  Set IR debounce value
 * @param  debounce: Number of shakes off,parameter range:0~255(Default 7) 
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRDebounce(uint8_t debounce)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x06, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = debounce;
  BM32S3021_1_sendBuf[5] = 28+debounce;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IR trigger threshold 
 * @param  threshold: Trigger threshold, parameter range:10~200(Default 16)  
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRThreshold(uint8_t threshold)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x07, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = threshold;
  BM32S3021_1_sendBuf[5] = 29+threshold;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IRQ triger time 
 * @param  irqTime: IRQ trigger time when gesture is valid, parameter range: 0~255 (Default 50) 
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others IRQ Triger Time =  irqTime*4ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRQTrigerTime(uint8_t irqTime)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x08, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = irqTime;
  BM32S3021_1_sendBuf[5] = 30+irqTime;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IR Cumulative continuous sliding time
 * @param  irTime: Cumulative continuous sliding time, parameter range:0~127(Default 30)
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others Cumulative continuous sliding time =  irTime*64ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRContinutyGestureTime(uint8_t irTime)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x09, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = irTime;
  BM32S3021_1_sendBuf[5] = 31+irTime;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IR fastest gesture time
 * @param  irTime: Fastest gesture to judge time, parameter range:0~200(Default 0)
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others Fastest gesture to judge time =  20+irTime*4ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRFastestGestureTime(uint8_t irTime)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x0A, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = irTime;
  BM32S3021_1_sendBuf[5] = 32+irTime;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IR Slowest gesture time
 * @param  irTime: Slowest gesture to judge time, parameter range:0~200(Default 80)
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others Slowest gesture to judge time = irTime*16ms
 ************************************************************************************************************/
uint8_t BM32S3021_1_setIRSlowestGestureTime(uint8_t irTime)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x0B, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = irTime;
  BM32S3021_1_sendBuf[5] = 33+irTime;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Write the version information low byte
 * @param  verl: version information low byte, parameter range:0x00~0xff
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others If the low byte is set to 0xaa, you can set the Current and OPA
 ************************************************************************************************************/
uint8_t _BM32S3021_1_writeVerL(uint8_t verl)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x00, 0x01, 0x00, 0X00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = verl;
  BM32S3021_1_sendBuf[5] = 22+verl;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Read IR1 reference
 * @param  void
* @retval BM32S3021_1_ref: IR1 reference
 ************************************************************************************************************/
uint8_t _BM32S3021_1_readIR1Ref(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x04, 0x01, 0xDA};
    uint8_t BM32S3021_1_ref = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_ref = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_ref;
}

/*********************************************************************************************************//**
 * @brief  Read IR2 reference
 * @param  void
 * @retval BM32S3021_1_ref: IR2 reference
 ************************************************************************************************************/
uint8_t _BM32S3021_1_readIR2Ref(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x05, 0x01, 0xDB};
    uint8_t BM32S3021_1_ref = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_ref = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_ref;
}

/*********************************************************************************************************//**
 * @brief  Read IRStatus, IRGerstureNum, IR1Ref, IR2Ref value
 * @param  buff[]: Stores the IR partial state parameter, parameter range: The minimum array length is 4 bytes  
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others   buff[0] : IR Status  
             buff[1] : IR Gersture Num
             buff[2] : IR1 Ref 
             buff[3] : IR2 Ref
 ************************************************************************************************************/
uint8_t _BM32S3021_1_readIrA2_A5(uint8_t  buff[])
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x02, 0x04, 0xDB};
    uint8_t BM32S3021_1_buff[9] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,9,100)== BM32S3021_1_CHECK_OK)
    {
     buff[0] = BM32S3021_1_buff[4];
     buff[1] = BM32S3021_1_buff[5];
     buff[2] = BM32S3021_1_buff[6];
     buff[3] = BM32S3021_1_buff[7];
			_BM32S3021_1_delay(20);
		 return BM32S3021_1_R_SUCCESS;
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  Read IRDebounce,IRThreshold,IRQTrigerTime,ContinutyGerstureTime,FastestGerstureTime,SlowestGerstureTime value
 * @param  buff[]: Stores the IR partial state parameter, parameter range: The minimum array length is 6 bytes  
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others   buff[0] : IR Debounce  
             buff[1] : IR Threshold
             buff[2] : IRQ Triger Time 
             buff[3] : Continuty Gersture Time
             buff[4] : Fastest Gersture Time 
             buff[5] : Slowest Gersture Time  
 ************************************************************************************************************/
uint8_t _BM32S3021_1_readIrA6_Ab(uint8_t  buff[])
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x06, 0x06, 0xE1};
    uint8_t BM32S3021_1_buff[11] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,11,100)== BM32S3021_1_CHECK_OK)
    {
     buff[0] = BM32S3021_1_buff[4];
     buff[1] = BM32S3021_1_buff[5];
     buff[2] = BM32S3021_1_buff[6];
     buff[3] = BM32S3021_1_buff[7];
		 buff[4] = BM32S3021_1_buff[8];
     buff[5] = BM32S3021_1_buff[9];
			_BM32S3021_1_delay(20);
		 return BM32S3021_1_R_SUCCESS;
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_R_FAIL;
}


/*********************************************************************************************************//**
 * @brief  Get IR1 emission current value
 * @param  void
 * @retval BM32S3021_1_Current: IR1 emission current value,Unit (mA)
 ************************************************************************************************************/
uint8_t _BM32S3021_1_getIR1Current(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x22, 0x01, 0xF8};
    uint8_t BM32S3021_1_Current = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_Current = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_Current;
}

/*********************************************************************************************************//**
 * @brief  Get IR2 emission current value
 * @param  void
 * @retval BM32S3021_1_Current: IR2 emission current value,Unit (mA)
 ************************************************************************************************************/
uint8_t _BM32S3021_1_getIR2Current(void)
{
    uint8_t BM32S3021_1_sendBuf[5] = {0x55, 0x80, 0x23, 0x01, 0xF9};
    uint8_t BM32S3021_1_Current = 0;
    uint8_t BM32S3021_1_buff[6] = {0};
    _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,5);
    if(_BM32S3021_1_readBytes(BM32S3021_1_buff,6,100)== BM32S3021_1_CHECK_OK)
    {
     BM32S3021_1_Current = BM32S3021_1_buff[4];
    }
    _BM32S3021_1_delay(20);
    return BM32S3021_1_Current;
}


/*********************************************************************************************************//**
 * @brief  Set IR1 emission current value
 * @param  current: IR1 emission current value, parameter range:0~31(Default 25)
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others Version information Low bytes You can set the OPA value only when this parameter is set to 0Xaa
 ************************************************************************************************************/
uint8_t _BM32S3021_1_setIR1Current(uint8_t current)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x22, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = current;
  BM32S3021_1_sendBuf[5] = 56+current;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  Set IR1 emission current value
 * @param  current: IR1 emission current value, parameter range:0~31(Default 25)
 * @retval Implementation status: 
            @arg 0(BM32S3021_1_R_SUCCESS): Success   
            @arg 1(BM32S3021_1_R_FAIL): Fail
 * @Others Version information Low bytes You can set the OPA value only when this parameter is set to 0Xaa
 ************************************************************************************************************/
uint8_t _BM32S3021_1_setIR2Current(uint8_t current)
{
  uint8_t BM32S3021_1_sendBuf[6] = {0x55, 0xC0, 0x23, 0x01, 0x00, 0x00};
  uint8_t BM32S3021_1_buff[3] = {0};
  BM32S3021_1_sendBuf[4] = current;
  BM32S3021_1_sendBuf[5] = 57+current;
  _BM32S3021_1_writeBytes(BM32S3021_1_sendBuf,6);
  if(_BM32S3021_1_readBytes(BM32S3021_1_buff,3,100)== BM32S3021_1_CHECK_OK)
   {
    if(BM32S3021_1_buff[1]== 0x7f)
    {
     _BM32S3021_1_delay(20);
     return BM32S3021_1_R_SUCCESS;
    }
   }
   _BM32S3021_1_delay(20);
   return BM32S3021_1_R_FAIL ;
}

/*********************************************************************************************************//**
 * @brief  write data through UART.
 * @param  wbuf:Variables for storing Data to be sent
 * @param  wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void _BM32S3021_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	 UARTM_DiscardReadBuffer(gBM32S3021_1_SERIAL);
	 UARTM_Write(gBM32S3021_1_SERIAL,wbuf,wlen);
}

/*********************************************************************************************************//**
 * @brief  read data through UART.
 * @param  rbuf:Variables for storing Data to be obtained
 * @param  rlen:Length of data to be obtained
 * @param  timOut:The timeout between reading the first byte and each byte,Unit (ms)
 * @retval BM32S3021_1_TIMEOUT_ERROR/BM32S3021_1_CHECK_OK/BM32S3021_1_CHECK_ERROR
 ************************************************************************************************************/
uint8_t _BM32S3021_1_readBytes(uint8_t rbuf[], uint8_t rlen, uint16_t timOut)
{
	  uint8_t _lenCount = 0;
	  uint8_t _delayCount = 0;
	  uint8_t _BM32S3021_1_checkSum = 0;
		 for(_lenCount = 0; _lenCount < rlen; _lenCount++)
		 {
			 _delayCount = 0;
			 while(UARTM_GetReadBufferLength(gBM32S3021_1_SERIAL) == 0)
			 {
				 if(_delayCount > timOut)
				 {
					 return BM32S3021_1_TIMEOUT_ERROR;
				 }
				 
				 _BM32S3021_1_delay(1);
				 _delayCount++;
			 }
			 UARTM_ReadByte(gBM32S3021_1_SERIAL,rbuf+_lenCount);		 
		 }

	 //checkSum
  for (_lenCount = 0; _lenCount < (rlen - 1); _lenCount++)
 {
    _BM32S3021_1_checkSum += rbuf[_lenCount];
  }
  if (_BM32S3021_1_checkSum == rbuf[rlen - 1])
  {
    return BM32S3021_1_CHECK_OK; // Check correct
  }
  else
 {
   return BM32S3021_1_CHECK_ERROR; // Check error
  }
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count: Delay time,Unit: ms
 * @retval void
 ************************************************************************************************************/
void _BM32S3021_1_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

/* Private functions ---------------------------------------------------------------------------------------*/
