/*********************************************************************************************************//**
 * @file    BM25S2021_1_HT32/src/BM25S2021_1.c
 * @version V1.0.1
 * @date    2024-08-12
 * @brief   The function of BM25S2021_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM25S2021_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM25S2021_1_WIRE = BM25S2021_1_WIRE;
uint8_t BM25S2021_1_commChoice = BM25S2021_1_COMM_IIC;

uint8_t BM25S2021_1_t_Flag = 0;
uint8_t BM25S2021_1_h_Flag = 0;
uint8_t BM25S2021_1_cnt = 7;
uint8_t BM25S2021_1_idx = 0;
uint16_t BM25S2021_1_loopCnt;
uint8_t BM25S2021_1_dataBuff[12]={0};
uint8_t BM25S2021_1_bytes[5]={0};   //Array for storing data

vu32 BM25S2021_1_SYSTICK_DelayTime =0;
vu8  BM25S2021_1_timeFlag = 0;
vu32 BM25S2021_1_timeCount = 0;
/* Private variables ---------------------------------------------------------------------------------------*/
uint16_t  _BM25S2021_1_pid;
uint8_t   _BM25S2021_1_ver;
uint32_t  _BM25S2021_1_sn;
/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief module wire number select.
 * @param wire_number: select wire number.
 * @retval BM25S2021_1_SUCCESS/BM25S2021_1_FAILURE
 ************************************************************************************************************/
BM25S2021_1_selStatus BM25S2021_1_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BM25S2021_1_FAILURE;
  }

  gBM25S2021_1_WIRE = wire_number;
  return BM25S2021_1_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM25S2021_1_I2C_Init(void)
{
	BM25S2021_1_commChoice = BM25S2021_1_COMM_IIC;
	I2CMaster_Init(gBM25S2021_1_WIRE, BM25S2021_1_MASTERADDR, BM25S2021_1_CLKFREQ);
	_BM25S2021_1_SYSTICK_Init();
}

/*********************************************************************************************************//**
 * @brief  Module initialization using One-Wire communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM25S2021_1_OneWire_Init(void)
{
  BM25S2021_1_commChoice = BM25S2021_1_COMM_ONEWIRE;
	//OUTPUT/INTPUT:BM25S2021_1 Data Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BM25S2021_1_DATAPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BM25S2021_1_DATAPIN_GPIO_ID, BM25S2021_1_DATAPIN_AFIO_PIN, AFIO_FUN_GPIO);
	
	_BM25S2021_1_SYSTICK_Init();
	
}

/*********************************************************************************************************//**
 * @brief  readTemperature
 * @param  isFahrenheit
            @arg true:Fahrenheit degree
            @arg false:centigrade(default)
 * @retval temperature , unit:'C (default) or 'F
 ************************************************************************************************************/
float BM25S2021_1_readTemperature(bool isFahrenheit)
{
    float BM25S2021_1_temperature = 0;
    /****** if select I2C communication ******/
    if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)     
    {    
       uint8_t BM25S2021_1_sendbuf[3] = {3,2,2};
       BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
       _BM25S2021_1_delay_ms(2);
       if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x06) == 0)
       {
          BM25S2021_1_temperature = (BM25S2021_1_dataBuff[2]<<8)|BM25S2021_1_dataBuff[3];
          if(isFahrenheit == true)
          {
             BM25S2021_1_temperature = BM25S2021_1_temperature*0.18 + 32;  
          }
          else
          {
             BM25S2021_1_temperature = BM25S2021_1_temperature / 10;
          }
          return BM25S2021_1_temperature;
       }
       else
       {
         return 0;//READ_ERROR
       }
    }
    /****** if select ONEWIRE communication ******/
    else
    {
      if(BM25S2021_1_h_Flag == 0)      //read temperature for the first time
      {
        BM25S2021_1_t_Flag = 1;
        BM25S2021_1_readOneWireData();  //read data
      }
      else
      {
        BM25S2021_1_h_Flag = 0;
      }
      if(BM25S2021_1_checkSum() == BM25S2021_1_READ_OK)
      {
         BM25S2021_1_temperature = (BM25S2021_1_bytes[2]<<8) | BM25S2021_1_bytes[3];
         if(isFahrenheit == true)
         {
            BM25S2021_1_temperature = BM25S2021_1_temperature*0.18 + 32;  
         }
         else BM25S2021_1_temperature = BM25S2021_1_temperature / 10;
         return BM25S2021_1_temperature;//READ_OK
      }
      else return 0;//READ_ERROR
    }
}

/*********************************************************************************************************//**
 * @brief  readHumidity
 * @param  void
 * @retval humidity,uint:%
 ************************************************************************************************************/
float BM25S2021_1_readHumidity(void)
{
    float BM25S2021_1_humidity = 0;
    /****** if select I2C communication ******/
    if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)
    {
        uint8_t BM25S2021_1_sendbuf[3] = {3,0,2};
        BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
        _BM25S2021_1_delay_ms(2);
        if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x06) == 0)
        {
          BM25S2021_1_humidity = (BM25S2021_1_dataBuff[2]<<8)|BM25S2021_1_dataBuff[3];
          BM25S2021_1_humidity = BM25S2021_1_humidity / 10;
          return BM25S2021_1_humidity; //READ_OK
        }
        else return 0;//READ_ERROR
    }
    /****** if select ONEWIRE communication ******/
    else
    {
      if(BM25S2021_1_t_Flag == 0)      //read humidity for the first time
      {
        BM25S2021_1_h_Flag = 1;
        BM25S2021_1_readOneWireData();  //read data
      }
      else
      {
        BM25S2021_1_t_Flag = 0; 
      }
      if(BM25S2021_1_checkSum() == BM25S2021_1_READ_OK)
      {
         BM25S2021_1_humidity = (BM25S2021_1_bytes[0]<<8) | BM25S2021_1_bytes[1];
         BM25S2021_1_humidity = BM25S2021_1_humidity / 10;            
         return BM25S2021_1_humidity;//READ_OK
      }
      else return 0;//READ_ERROR
   }
}

/*********************************************************************************************************//**
 * @brief  Get device Ver
 * @param  void
 * @retval Ver 
 ************************************************************************************************************/
uint8_t  BM25S2021_1_getFWVer(void)
{
    uint8_t BM25S2021_1_sendbuf[3] = {3,8,7};
    BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
    _BM25S2021_1_delay_ms(2);
    if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x0b) == 0)
    {
       _BM25S2021_1_ver = BM25S2021_1_dataBuff[4];
       return _BM25S2021_1_ver;
    }
    else return 0; //READ_ERROR
}

/*********************************************************************************************************//**
 * @brief  Get device PID
 * @param  void
 * @retval PID 
 ************************************************************************************************************/
uint16_t BM25S2021_1_getPID(void)
{
    uint8_t BM25S2021_1_sendbuf[3] = {3,8,7};
    BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
    _BM25S2021_1_delay_ms(2);
    if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x0b) == 0)
    {
       _BM25S2021_1_pid = (BM25S2021_1_dataBuff[2]<<8)|BM25S2021_1_dataBuff[3];
       return _BM25S2021_1_pid;
    }
    else return 0; //READ_ERROR
}

/*********************************************************************************************************//**
 * @brief  Get device SN
 * @param  void
 * @retval SN
 ************************************************************************************************************/
uint32_t BM25S2021_1_getSN(void)
{
    uint8_t BM25S2021_1_sendbuf[3] = {3,8,7};
    BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
    _BM25S2021_1_delay_ms(2);
    if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x0b) == 0)
    {
       _BM25S2021_1_sn  = ((long)BM25S2021_1_dataBuff[5]<<24)|((long)BM25S2021_1_dataBuff[6]<<16)|((long)BM25S2021_1_dataBuff[7]<<8)|BM25S2021_1_dataBuff[8];
       return _BM25S2021_1_sn;
    }
    else return 0; //READ_ERROR
}



/* Private functions ---------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief  read Temperature RawData
 * @param  rawTemperature:Variables for storing temperature RawDa
 * @retval temperature Raw Data(uint16_t)     
 ************************************************************************************************************/
uint16_t BM25S2021_1_readRawTemperature(void)
{
    uint16_t BM25S2021_1_temperature = 0;
    /****** if select I2C communication ******/
    if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)     
    {    
       uint8_t BM25S2021_1_sendbuf[3] = {3,2,2};
       BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
       _BM25S2021_1_delay_ms(2);
       if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x06) == 0)
       {
          BM25S2021_1_temperature = (BM25S2021_1_dataBuff[2]<<8)|BM25S2021_1_dataBuff[3];
          return BM25S2021_1_temperature;
       }
       else
       {
         return 0;//READ_ERROR
       }
    }
    /****** if select ONEWIRE communication ******/
    else
    {
      if(BM25S2021_1_h_Flag == 0)
      {
        BM25S2021_1_t_Flag = 1;
        BM25S2021_1_readOneWireData();
      }
      else
      {
        BM25S2021_1_h_Flag = 0;
      }
      if(BM25S2021_1_checkSum() == BM25S2021_1_READ_OK)
      {
         BM25S2021_1_temperature = (BM25S2021_1_bytes[2]<<8) | BM25S2021_1_bytes[3];
         return BM25S2021_1_temperature;//READ_OK
      }
      else return 0;//READ_ERROR
    }
}

/*********************************************************************************************************//**
 * @brief  read Humidity RawData
 * @param  rawHumidity:Variables for storing humidity  Rawdata
 * @retval humidity  Raw data(uint16_t) 
 ************************************************************************************************************/
uint16_t BM25S2021_1_readRawHumidity(void)
{
    uint16_t BM25S2021_1_humidity = 0;
    /****** if select I2C communication ******/
    if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)
    {
        uint8_t BM25S2021_1_sendbuf[3] = {3,0,2};
        BM25S2021_1_writeBytes(BM25S2021_1_sendbuf, 0x03);
        _BM25S2021_1_delay_ms(2);
        if(BM25S2021_1_readBytes(BM25S2021_1_dataBuff,0x06) == 0)
        {
          BM25S2021_1_humidity = (BM25S2021_1_dataBuff[2]<<8)|BM25S2021_1_dataBuff[3];
          return BM25S2021_1_humidity; //READ_OK
        }
        else return 0;//READ_ERROR
    }
    /****** if select ONEWIRE communication ******/
    else
    {
      if(BM25S2021_1_t_Flag == 0)
      {
        BM25S2021_1_h_Flag = 1;
        BM25S2021_1_readOneWireData();
      }
      else
      {
        BM25S2021_1_t_Flag = 0;
      }
      if(BM25S2021_1_checkSum() == BM25S2021_1_READ_OK)
      {
         BM25S2021_1_humidity = (BM25S2021_1_bytes[0]<<8) | BM25S2021_1_bytes[1];         
         return BM25S2021_1_humidity;//READ_OK
      }
      else return 0;//READ_ERROR
   }
}

/*********************************************************************************************************//**
 * @brief  Clear data buff
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM25S2021_1_clearBuff(void)
{
	int BM25S2021_1_a = 0;
  if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)
  {
      for(BM25S2021_1_a = 0; BM25S2021_1_a < 12; BM25S2021_1_a++)
      {
        BM25S2021_1_dataBuff[BM25S2021_1_a] = 0;
      } 
  }
  else
  {
      for(BM25S2021_1_a = 0; BM25S2021_1_a < 5; BM25S2021_1_a++)
      {
        BM25S2021_1_bytes[BM25S2021_1_a] = 0;
      } 
  } 
}

   
/*IIC*/
/*********************************************************************************************************//**
 * @brief  writeData
 * @param  wbuf:Variables for storing Data to be sent
           wlen:Length of data sent
 * @retval void
 ************************************************************************************************************/
void BM25S2021_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)
  {
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = BM25S2021_1_MASTERADDR;
    gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;

    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBM25S2021_1_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBM25S2021_1_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
	}
}

/*********************************************************************************************************//**
 * @brief  readData
 * @param  rbuf:Variables for storing Data to be obtained
           rlen:Length of data to be obtained
 * @retval BM25S2021_1_READ_OK/BM25S2021_1_READ_ERROR  
 ************************************************************************************************************/
uint8_t BM25S2021_1_readBytes(uint8_t rbuf[], uint8_t rlen)
{
    if(BM25S2021_1_commChoice == BM25S2021_1_COMM_IIC)
    {
       BM25S2021_1_clearBuff();
			{
       I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
       I2CMaster_Typedef I2CMaster = { 0 };

       I2CMaster.uSlaveAddress = BM25S2021_1_MASTERADDR;
       I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
       I2CMaster.Rx.uLength = rlen;
       I2CMaster.uTimeout_us = 30000;
       I2CMaster.isDynamicOnOff_I2C = FALSE;

       I2CMaster_Read(gBM25S2021_1_WIRE, &I2CMaster);
	
       do {
         I2CResult = I2CMaster_GetStatus(gBM25S2021_1_WIRE);
       } while (I2CResult == I2C_MASTER_GOING);
		  }
       /* Check Sum */
       if(rbuf[0] != 0x03)
       {
         return BM25S2021_1_READ_ERROR;
       }
       if(BM25S2021_1_cheackCRC(rbuf,rlen) == 0)
       {
         return BM25S2021_1_READ_OK; //READ_OK
       }
    }
    return 0x00;
}

/*********************************************************************************************************//**
 * @brief  CRC Correction
 * @param  *buff:Data obtained through IIC
           len :Gets the total length of the data
 * @retval Checksum calculate by CRC-16
 ************************************************************************************************************/
uint8_t BM25S2021_1_cheackCRC(uint8_t *buff,uint8_t len)
{
  uint16_t BM25S2021_1_temp_crc=0xffff;
  uint8_t BM25S2021_1_i;
  while(len--)
  {
    BM25S2021_1_temp_crc ^= *buff++;
    for(BM25S2021_1_i=0;BM25S2021_1_i<8;BM25S2021_1_i++)
    {
      if(BM25S2021_1_temp_crc & 0x01)
      {
        BM25S2021_1_temp_crc >>= 1;
        BM25S2021_1_temp_crc ^= 0xa001;
      }
      else
      {
        BM25S2021_1_temp_crc >>=1;
      }
    }
  }
  return BM25S2021_1_temp_crc;
}
   
/*One Wire*/
/*********************************************************************************************************//**
 * @brief  readOneWireData
 * @param  void
 * @retval BM25S2021_1_READ_OK/BM25S2021_1_ERROR_TIMEOUT
 ************************************************************************************************************/
uint8_t BM25S2021_1_readOneWireData(void)
{
	  int BM25S2021_1_i = 0;

    /*****   start    ******/
	   GPIO_DirectionConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, GPIO_DIR_OUT);  //OUTPUT
     GPIO_WriteOutBits(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, RESET); //OUTPUT LOW
     _BM25S2021_1_delay_ms(15);
     GPIO_WriteOutBits(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, SET); //OUTPUT HIGH
	   GPIO_DirectionConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, GPIO_DIR_IN);//INPUT
	   GPIO_InputConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, ENABLE);//INPUT ENABLE
     BM25S2021_1_loopCnt = 10000;  //10ms
     while (GPIO_ReadInBit(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN) == 1)
     {
       if (BM25S2021_1_loopCnt-- == 0)
       {
         BM25S2021_1_clearBuff();
         return BM25S2021_1_ERROR_TIMEOUT;
       }
     }

    /*****   ACK    ******/
	   GPIO_DirectionConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, GPIO_DIR_IN);//INPUT
	   GPIO_InputConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, ENABLE);//INPUT ENABLE
     BM25S2021_1_loopCnt = 1000;   //1ms,100 * delay_10us
     while (GPIO_ReadInBit(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN) == 0)
     {
       if (BM25S2021_1_loopCnt-- == 0) return BM25S2021_1_ERROR_TIMEOUT;
     }
     BM25S2021_1_loopCnt = 1000;  //1ms,100 * delay_10us
     while (GPIO_ReadInBit(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN) == 1)
     {
       if (BM25S2021_1_loopCnt-- == 0) return BM25S2021_1_ERROR_TIMEOUT;
     }

    /*****   readData    ******/
     BM25S2021_1_clearBuff();
     BM25S2021_1_cnt = 7;
     BM25S2021_1_idx = 0; 
     for (BM25S2021_1_i = 0; BM25S2021_1_i < 40; BM25S2021_1_i++)
     {  
          while (GPIO_ReadInBit(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN) == 0);
			    BM25S2021_1_timeCount = 0;
			    BM25S2021_1_timeFlag = 1; //Start timing
          
          while (GPIO_ReadInBit(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN) == 1);
					BM25S2021_1_timeFlag = 0;  //Stop timing
          if (BM25S2021_1_timeCount >= 4) //>40us
					{
						BM25S2021_1_bytes[BM25S2021_1_idx] |= (1 << BM25S2021_1_cnt);
						
					}
          BM25S2021_1_timeCount = 0;  
          if (BM25S2021_1_cnt == 0)   // next byte?
          {
            BM25S2021_1_cnt = 7;    // restart at MSB
            BM25S2021_1_idx++;      // next byte!
          }
          else BM25S2021_1_cnt--;
          if(BM25S2021_1_idx > 4) BM25S2021_1_idx = 0;
     }
     GPIO_DirectionConfig(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, GPIO_DIR_OUT);  //OUTPUT
     GPIO_WriteOutBits(BM25S2021_1_DATAPIN_GPIO_PORT, BM25S2021_1_DATAPIN_GPIO_PIN, SET); //OUTPUT HIGH
     return BM25S2021_1_READ_OK; 
}

/*********************************************************************************************************//**
 * @brief  check Sum
 * @param  void
 * @retval BM25S2021_1_READ_OK/BM25S2021_1_ERROR_CHECKSUM 
 ************************************************************************************************************/
int BM25S2021_1_checkSum(void)
{
	    uint8_t BM25S2021_1_sum = 0;
      BM25S2021_1_sum = BM25S2021_1_bytes[0] + BM25S2021_1_bytes[1] + BM25S2021_1_bytes[2] + BM25S2021_1_bytes[3];
      if(BM25S2021_1_bytes[4] != BM25S2021_1_sum) return BM25S2021_1_ERROR_CHECKSUM;
      else return BM25S2021_1_READ_OK;
}


/*********************************************************************************************************//**
 * @brief  SYSTICK  Config 
 * @param  void
 * @retval void
 ************************************************************************************************************/
void _BM25S2021_1_SYSTICK_Init(void)
{
  /* SYSTICK configuration */
  SYSTICK_ClockSourceConfig(SYSTICK_SRC_STCLK);          // Default : CK_AHB/8
  SYSTICK_SetReloadValue(SystemCoreClock / 8 / 100000);  // (CK_AHB/8/100000) = 10us on chip
  SYSTICK_IntConfig(ENABLE);                             // Enable SYSTICK Interrupt
  SYSTICK_CounterCmd(SYSTICK_COUNTER_CLEAR);
  SYSTICK_CounterCmd(SYSTICK_COUNTER_ENABLE);
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BM25S2021_1_delay_ms(vu32 count)
{
	u32 BM25S2021_1_i;
	for(BM25S2021_1_i=0;BM25S2021_1_i<count;BM25S2021_1_i++)
	{
		_BM25S2021_1_delay_10us(100);
	}
}

/*********************************************************************************************************//**
 * @brief  delay 10us.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BM25S2021_1_delay_10us(vu32 count)
{
  BM25S2021_1_SYSTICK_DelayTime = count;
  while(BM25S2021_1_SYSTICK_DelayTime != 0);
}

