/*********************************************************************************************************//**
 * @file    BMS81M001_HT32/src/BMS81M001.c
 * @version V1.0.1
 * @date    2024-08-26
 * @brief   The function of BMS81M001 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMS81M001.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMS81M001_WIRE = BMS81M001_WIRE;
uint32_t gBMS81M001_i2cAddress = 0x43;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief  module wire number select.
 * @param  wire_number: select wire number.
 * @retval BMS81M001_selStatus
                    @arg BMS81M001_SUCCESS
                    @arg BMS81M001_FAILURE 
 ************************************************************************************************************/
BMS81M001_selStatus BMS81M001_selWire(uint32_t wire_number)
{
  if(CHECK_SERIALn(wire_number) == -1)
  {
    return BMS81M001_FAILURE;
  }

  gBMS81M001_WIRE = wire_number;
  return BMS81M001_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using IIC communication.
 * @param  void
 * @retval void       
 ************************************************************************************************************/
void BMS81M001_Init(void)
{
	I2CMaster_Init(gBMS81M001_WIRE, gBMS81M001_i2cAddress, BMS81M001_CLKFREQ);
	//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BMS81M001_STATUSPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BMS81M001_STATUSPIN_GPIO_ID, BMS81M001_STATUSPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BMS81M001_STATUSPIN_GPIO_PORT, BMS81M001_STATUSPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BMS81M001_STATUSPIN_GPIO_PORT, BMS81M001_STATUSPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BMS81M001_STATUSPIN_GPIO_PORT, BMS81M001_STATUSPIN_GPIO_PIN, ENABLE);
	//
	_BMS81M001_wakeup();
}


/*********************************************************************************************************//**
 * @brief  Get STA PIN Status
 * @param  void
 * @retval STA PIN Status:
                    @arg 0:INT output low level 
                    @arg 1:INT output high level 
 ************************************************************************************************************/
uint8_t BMS81M001_getSTATUS(void)
{
	return (GPIO_ReadInBit(BMS81M001_STATUSPIN_GPIO_PORT, BMS81M001_STATUSPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  set motion threshold
 * @param  
           @arg thr:Vibration threshold,Parameter range:1~255, unit: : 1LSB/1mg
           @arg dur:duration,Parameter range:1~255, unit: 1LSB/1ms 
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t BMS81M001_setShake(uint8_t thr,uint8_t dur)
{
  uint8_t BMS81M001_sendBuf[5] = {0x04,0x01,0x00,0x00,0x00};
	uint8_t BMS81M001_buff[4] = {0};
	if(thr<1)
  {
    thr = 1; 
  }
  if(dur<1)
  {
    dur = 1; 
  }
	BMS81M001_sendBuf[2] = thr;
	BMS81M001_sendBuf[3] = dur;
	BMS81M001_sendBuf[4] = ~(BMS81M001_sendBuf[0]+BMS81M001_sendBuf[1]+BMS81M001_sendBuf[2]+BMS81M001_sendBuf[3]);
  _BMS81M001_writeBytes(BMS81M001_sendBuf,5); 
	_BMS81M001_delay(2);
  if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
   if(BMS81M001_buff[2] == 0x01)
    {
     return BMS81M001_R_SUCCESS;
    }
	}
	 return BMS81M001_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  Set the time for entering the low-power mode
 * @param  
           @arg  idle_delay=0: Module always working
           @arg  idle_delay>0: Time to enter low power mode,Parameter range:1~255, unit:s 
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t BMS81M001_setIdleModeDelay(uint8_t idle_delay)
{
  uint8_t BMS81M001_sendBuf[4] = {0x03,0x02,0x00,0x00};
	uint8_t BMS81M001_buff[4] = {0};
	BMS81M001_sendBuf[2] = idle_delay;
	BMS81M001_sendBuf[3] = ~(BMS81M001_sendBuf[0]+BMS81M001_sendBuf[1]+BMS81M001_sendBuf[2]);
  _BMS81M001_writeBytes(BMS81M001_sendBuf,4); 
	_BMS81M001_delay(2);
  if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
   if(BMS81M001_buff[2] == 0x01)
    {
     return BMS81M001_R_SUCCESS;
    }
	}
	 return BMS81M001_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  Set work mode
 * @param 
           @arg rate:work mode
                 0:Normal mode
                 2:Low power mode 1
                 3:Low power mode 2
                 4:Low power mode 3
          @arg isEnable == true: Working mode setting Enabled, isEnable == false: Working mode setting Disable
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t BMS81M001_setWorkMode(uint8_t rate,bool isEnable)
{
  uint8_t BMS81M001_sendBuf[4] = {0x03,0x06,0x00,0x00};
	uint8_t BMS81M001_buff[4] = {0};
	if(rate>4)
  {
   rate = 0x00;
  }
	if(isEnable == true)
  {
   BMS81M001_sendBuf[2]=rate;
  }
  else
  {
   BMS81M001_sendBuf[2] = 0x00;
  }
	BMS81M001_sendBuf[3] = ~(BMS81M001_sendBuf[0]+BMS81M001_sendBuf[1]+BMS81M001_sendBuf[2]);
  _BMS81M001_writeBytes(BMS81M001_sendBuf,4); 
	_BMS81M001_delay(2);
  if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
   if(BMS81M001_buff[2] == 0x01)
    {
     return BMS81M001_R_SUCCESS;
    }
	}
	 return BMS81M001_R_FAIL;

}

/*********************************************************************************************************//**
 * @brief  get vibration status
 * @param  void
 * @retval BMS81M001_ShakeStatus: 
             @arg 0: No vibration detected
             @arg 1: Vibration detected
 ************************************************************************************************************/
uint8_t BMS81M001_getShakeStatus(void)
{
  uint8_t BMS81M001_sendBuf[3] = {0x02,0x03,0xfa};
	uint8_t BMS81M001_buff[4] = {0};
	uint8_t BMS81M001_ShakeStatus;
  _BMS81M001_writeBytes(BMS81M001_sendBuf,3); 
	_BMS81M001_delay(2);
  if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
    BMS81M001_ShakeStatus = BMS81M001_buff[2];
	}
	 return BMS81M001_ShakeStatus;
}

/*********************************************************************************************************//**
 * @brief  get parameter setting
 * @param   
           @arg *thr: Store the shock threshold
           @arg *dur: Store the duration
           @arg *idle_delay: Store the Time to enter low power mode
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t BMS81M001_getParameterSetting(uint8_t *thr,uint8_t *dur,uint8_t *idle_delay)
{
  uint8_t BMS81M001_sendBuf[3] = {0x02,0x05,0xf8};
	uint8_t BMS81M001_buff[6] = {0};
  _BMS81M001_writeBytes(BMS81M001_sendBuf,3); 
	_BMS81M001_delay(2);
  if(_BMS81M001_readBytes(BMS81M001_buff,6) == BMS81M001_CHECK_OK)
	{
    *thr = BMS81M001_buff[2];
    *dur = BMS81M001_buff[3];
    *idle_delay = BMS81M001_buff[4];
		return BMS81M001_R_SUCCESS;
	}
	 return BMS81M001_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  get Fermware version
 * @param  void
 * @retval BMS81M001_FWVer: Fermware version
 ************************************************************************************************************/
uint16_t BMS81M001_getFWVer(void)
{
  uint8_t BMS81M001_sendBuf[3] = {0x02,0x04,0xf9};
	uint8_t BMS81M001_buff[5] = {0};
	uint16_t BMS81M001_FWVer;
  _BMS81M001_writeBytes(BMS81M001_sendBuf,3); 
	_BMS81M001_delay(2); 
  if(_BMS81M001_readBytes(BMS81M001_buff,5) == BMS81M001_CHECK_OK)
	{  
		BMS81M001_FWVer =  ((uint16_t)BMS81M001_buff[2])<<8 | (uint16_t)BMS81M001_buff[3];
	}
	 return BMS81M001_FWVer;
}

/*********************************************************************************************************//**
 * @brief  mode reset
 * @param  void
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t BMS81M001_reset(void)
{
  uint8_t BMS81M001_sendBuf[3] = {0x02,0x00,0xfd};
	uint8_t BMS81M001_buff[4] = {0};
  _BMS81M001_writeBytes(BMS81M001_sendBuf,3); 
	_BMS81M001_delay(2);
   if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
   if(BMS81M001_buff[2] == 0x01)
    {
     return BMS81M001_R_SUCCESS;
    }
	}
	 return BMS81M001_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  wakeup
 * @param  void
 * @retval Implementation status
           @arg 0(BMS81M001_R_SUCCESS): SUCCESS
           @arg 1(BMS81M001_R_FAIL): FAIL
 ************************************************************************************************************/
uint8_t _BMS81M001_wakeup(void)
{
  uint8_t BMS81M001_sendBuf[3] = {0x02,0x07,0xf6};
	uint8_t BMS81M001_buff[4] = {0};
  _BMS81M001_writeBytes(BMS81M001_sendBuf,3); 
	_BMS81M001_delay(2);
   if(_BMS81M001_readBytes(BMS81M001_buff,4) == BMS81M001_CHECK_OK)
	{
   _BMS81M001_delay(250);
		if(BMS81M001_buff[2] == 0x01)
    {
     return BMS81M001_R_SUCCESS;
    }
	}
	 return BMS81M001_R_FAIL;
}

/*********************************************************************************************************//**
 * @brief  write data through IIC.
 * @param  wbuf:Variables for storing Data to be sent
 * @param  wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void _BMS81M001_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = gBMS81M001_i2cAddress;
    gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;
    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBMS81M001_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBMS81M001_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
}

/*********************************************************************************************************//**
 * @brief  read data through IIC.
 * @param  rbuf:Variables for storing Data to be obtained
 * @param  rlen:Length of data to be obtained
 * @retval void
 ************************************************************************************************************/
uint8_t _BMS81M001_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	 uint8_t _lenCount = 0;
	 uint8_t _BMS81M001_checkSum = 0;
   I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
   I2CMaster_Typedef I2CMaster = { 0 };

   I2CMaster.uSlaveAddress = gBMS81M001_i2cAddress;
   I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
   I2CMaster.Rx.uLength = rlen;
   I2CMaster.uTimeout_us = 30000;
   I2CMaster.isDynamicOnOff_I2C = FALSE;

   I2CMaster_Read(gBMS81M001_WIRE, &I2CMaster);
	
   do {
     I2CResult = I2CMaster_GetStatus(gBMS81M001_WIRE);
   } while (I2CResult == I2C_MASTER_GOING);
	 
	 //checkSum
  for (_lenCount = 0; _lenCount < (rlen - 1); _lenCount++)
 {
    _BMS81M001_checkSum += rbuf[_lenCount];
  }
  if ((uint8_t)(~_BMS81M001_checkSum) == rbuf[rlen - 1])
  {
    return BMS81M001_CHECK_OK; // Check correct
  }
  else
 {
   return BMS81M001_CHECK_ERROR; // Check error
  }
	
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  Delay time,Unit:ms
 * @retval void
 ************************************************************************************************************/
void _BMS81M001_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

/* Private functions ---------------------------------------------------------------------------------------*/
