/*********************************************************************************************************//**
 * @file    BM22S2021_1_HT32/src/BM22S2021_1.c
 * @version V1.0.1
 * @date    2025-06-19
 * @brief   The function of BM22S2021_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM22S2021_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM22S2021_1_SERIAL = BM22S2021_1_SERIAL;
uint8_t _BM22S2021_1_recBuf[41] = {0};
uint8_t _BM22S2021_1_autoTxMode=0x80;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief  module serial number select.
 * @param  serial_number: select serial number.
 * @retval BM22S2021_1_selStatus          
           @arg BM22S2021_1_SUCCESS
           @arg BM22S2021_1_FAILURE
 ************************************************************************************************************/
BM22S2021_1_selStatus BM22S2021_1_selSerial(uint32_t serial_number)
{
  if(CHECK_SERIALn(serial_number) == -1)
  {
    return BM22S2021_1_FAILURE;
  }

  gBM22S2021_1_SERIAL = serial_number;
  return BM22S2021_1_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using UART communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM22S2021_1_Init(void)
{
	{
	USART_InitTypeDef USART_InitStructure = {0};
  USART_InitStructure.USART_BaudRate = 9600;
  USART_InitStructure.USART_WordLength = USART_WORDLENGTH_8B;
  USART_InitStructure.USART_StopBits = USART_STOPBITS_1;
  USART_InitStructure.USART_Parity = USART_PARITY_NO;
  USART_InitStructure.USART_Mode = USART_MODE_NORMAL;
  UARTM_Init(gBM22S2021_1_SERIAL, &USART_InitStructure, BM22S2021_1_UART_TXTIMEOUT);
  }
		//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BM22S2021_1_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BM22S2021_1_INTPIN_GPIO_ID, BM22S2021_1_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BM22S2021_1_INTPIN_GPIO_PORT, BM22S2021_1_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BM22S2021_1_INTPIN_GPIO_PORT, BM22S2021_1_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BM22S2021_1_INTPIN_GPIO_PORT, BM22S2021_1_INTPIN_GPIO_PIN, ENABLE);
    
}


/*********************************************************************************************************//**
 * @brief  Get STA Pin Status
 * @param  void
 * @retval STA Pin Status:
                    @arg 0:STA Pin output low level 
                    @arg 1:STA Pin output high level 
 ************************************************************************************************************/
uint8_t BM22S2021_1_getSTATUS(void)
{
	return (GPIO_ReadInBit(BM22S2021_1_INTPIN_GPIO_PORT,BM22S2021_1_INTPIN_GPIO_PIN));
}


/*********************************************************************************************************//**
 * @brief  Query the FW version
 * @param  void
 * @retval FW version
 * @Others The FW version number and production date are both 8421 BCD code
 ************************************************************************************************************/
uint16_t BM22S2021_1_getFWVer(void)
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xAD, 0x00, 0x00, 0x53};
  uint8_t BM22S2021_1_ack[12];
  uint16_t BM22S2021_1_FWVer=0;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd,4);
  _BM22S2021_1_delay(50);
  if (_BM22S2021_1_readBytes(BM22S2021_1_ack,12,100) == 0x00 && BM22S2021_1_ack[4] == 0xAD)
  {
   BM22S2021_1_FWVer=((uint16_t)BM22S2021_1_ack[6]<<8 | BM22S2021_1_ack[7]);
  }
  _BM22S2021_1_delay(20);
  return  BM22S2021_1_FWVer;
}


/*********************************************************************************************************//**
 * @brief  Query production date.
 * @param  buff[]:3 byte
                    @arg buff[0]:year  
                    @arg buff[1]:month  
                    @arg buff[2]:day
 * @retval   1: module data acquisition failed, there is no correct feedback value
             0: Module data obtained successfully
 ************************************************************************************************************/
uint8_t BM22S2021_1_getProDate(uint8_t buff[])
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xAD, 0x00, 0x00, 0x53};
  uint8_t BM22S2021_1_ack[12];
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  if (_BM22S2021_1_readBytes(BM22S2021_1_ack,12,100) == 0x00 && BM22S2021_1_ack[4] == 0xAD)
  {
    buff[0]=BM22S2021_1_ack[8];
    buff[1]=BM22S2021_1_ack[9];
    buff[2]=BM22S2021_1_ack[10];
    _BM22S2021_1_delay(20);
    return    0;
  }
  else
  {
    _BM22S2021_1_delay(20);
    return   1;
  }
}


/*********************************************************************************************************//**
 * @brief  Read the current status and data of the equipment;
 * @param   buff:Store the read data //buff[40-0]/buff[20-0]
 * @retval   0: Read data successfully, module feedback is correct
             1: failed to read data, module feedback error
 ************************************************************************************************************/
uint8_t BM22S2021_1_requestInfoPackage(uint8_t buff[])
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xAC, 0x00, 0x00, 0x54};
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  if (_BM22S2021_1_readBytes(buff,41,100) == 0x00 && buff[4] == 0xAC)
  {
    _BM22S2021_1_delay(20);
    return    0;
  }
  else
  {
    _BM22S2021_1_delay(20);
    return   1;
  }
}

/*********************************************************************************************************//**
 * @brief  Determine whether it is the data packet sent by the module
 * @param   void
 * @retval   1:Have 41/21 data bytes
             0:no data
 ************************************************************************************************************/
bool BM22S2021_1_isInfoAvailable(void)
{
  uint8_t BM22S2021_1_recLen=41;
  uint8_t BM22S2021_1_header[5] = {0xAA,0x00, 0x11, 0x01, 0xAC};
  uint8_t BM22S2021_1_recBuf[41] = {0};
  uint8_t BM22S2021_1_i, BM22S2021_1_num = 0, BM22S2021_1_readCnt = 0, BM22S2021_1_failCnt = 0, BM22S2021_1_checkCode = 0;
  uint8_t BM22S2021_1_isHeader = 0; 
  bool  BM22S2021_1_result = false;
  uint8_t BM22S2021_1_rbuf[1];
 if(_BM22S2021_1_autoTxMode==0x80)
 {
  BM22S2021_1_recLen=41;
 }
 if(_BM22S2021_1_autoTxMode==0x81)
 {
  BM22S2021_1_recLen=21;
 }
  BM22S2021_1_header[1] = BM22S2021_1_recLen;
  BM22S2021_1_num = UARTM_GetReadBufferLength(gBM22S2021_1_SERIAL);
  /* Serial buffer contains at least one 32-byte data */
  if (BM22S2021_1_num >= BM22S2021_1_recLen)
  {
    while (BM22S2021_1_failCnt < 2) // Didn't read the required data twice, exiting the loop
    {
      /* Find 5-byte data header */
      for (BM22S2021_1_i = 0; BM22S2021_1_i < 5;)
      {
        UARTM_ReadByte(gBM22S2021_1_SERIAL,BM22S2021_1_rbuf);
        BM22S2021_1_recBuf[BM22S2021_1_i] = BM22S2021_1_rbuf[0];
        if (BM22S2021_1_recBuf[BM22S2021_1_i] == BM22S2021_1_header[BM22S2021_1_i])
        {
          BM22S2021_1_isHeader = 1; // Fixed code is correct
          BM22S2021_1_i++;             // Next byte
        }
        else if (BM22S2021_1_recBuf[BM22S2021_1_i] != BM22S2021_1_header[BM22S2021_1_i] && BM22S2021_1_i > 0)
        {
          BM22S2021_1_isHeader = 0; // Next fixed code error
          BM22S2021_1_failCnt++;
          break;
        }
        else if (BM22S2021_1_recBuf[BM22S2021_1_i] != BM22S2021_1_header[BM22S2021_1_i] && BM22S2021_1_i == 0)
        {
          BM22S2021_1_readCnt++; // 0xAA not found, continue
        }
        if (BM22S2021_1_readCnt >= (BM22S2021_1_num - 5))
        {
          BM22S2021_1_readCnt = 0;
          BM22S2021_1_isHeader = 0; //
          break;
        }
      }
      /* Find the correct data header */
      if (BM22S2021_1_isHeader)
      {
        for (BM22S2021_1_i = 0; BM22S2021_1_i < 5; BM22S2021_1_i++)
        {
          BM22S2021_1_checkCode += BM22S2021_1_recBuf[BM22S2021_1_i]; // Sum checkCode
        }
        for (BM22S2021_1_i = 5; BM22S2021_1_i < BM22S2021_1_recLen; BM22S2021_1_i++) // Read subsequent 27-byte data
        {
          UARTM_ReadByte(gBM22S2021_1_SERIAL,BM22S2021_1_rbuf);
          BM22S2021_1_recBuf[BM22S2021_1_i] = BM22S2021_1_rbuf[0];
          BM22S2021_1_checkCode += BM22S2021_1_recBuf[BM22S2021_1_i]; // Sum checkCode
        }
        BM22S2021_1_checkCode = BM22S2021_1_checkCode - BM22S2021_1_recBuf[BM22S2021_1_recLen - 1];
        BM22S2021_1_checkCode = (~BM22S2021_1_checkCode) + 1; // Calculate checkCode
        /* Compare whether the check code is correct */
        if (BM22S2021_1_checkCode == BM22S2021_1_recBuf[BM22S2021_1_recLen - 1])
        {
          for (BM22S2021_1_i = 0; BM22S2021_1_i < BM22S2021_1_recLen; BM22S2021_1_i++) // True, assign data to _recBuf[]
          {
            _BM22S2021_1_recBuf[BM22S2021_1_i] = BM22S2021_1_recBuf[BM22S2021_1_i];
          }
          BM22S2021_1_result = true;
          break; // Exit "while (failCnt < 2)" loop
        }
        else
        {
          BM22S2021_1_failCnt++; // Error, failCnt plus 1, return "while (failCnt < 2)" loop
          BM22S2021_1_checkCode = 0;
        }
      }
    }
    _BM22S2021_1_clear_UART_FIFO();
  }
  return BM22S2021_1_result;

}

/*********************************************************************************************************//**
 * @brief  Read the data automatically output by the module
 * @param  buff:41/21 byte
 * @retval   0: Read data successfully, module feedback is correct
             1: failed to read data, module feedback error
 ************************************************************************************************************/
void BM22S2021_1_readInfoPackage(uint8_t array[])
{
  uint8_t BM22S2021_1_recLen=41;
  uint8_t BM22S2021_1_i; 
 if(_BM22S2021_1_autoTxMode==0x80)
 {
  BM22S2021_1_recLen=41;
 }
 if(_BM22S2021_1_autoTxMode==0x81)
 {
  BM22S2021_1_recLen=21;
 }
  for (BM22S2021_1_i = 0; BM22S2021_1_i < BM22S2021_1_recLen; BM22S2021_1_i++)
  {
    array[BM22S2021_1_i] = _BM22S2021_1_recBuf[BM22S2021_1_i];
  }
}

/*********************************************************************************************************//**
 * @brief   Read the register of the specified address
 * @param   addr:Destination register address to read
            Register list:
                    @arg T0A upper / lower calibration limit (H+L):0X08  0X09
                    @arg T0B upper / lower calibration limit (H+L):0X0A  0X0B
                    @arg T0A alarm threshold(H+L):0X10 0X11
                    @arg T0B alarm threshold(H+L):0X12 0X13
                    @arg standby smoke detection cycle:0X2D
                    @arg serial port automatic output:0X2E
                    @arg alarm output level:0X2F
                    Data to be written
                    All available addresses have been defined in the H file   
 * @retval   data:Store read data
             0:read error
 ************************************************************************************************************/
uint8_t BM22S2021_1_readRegister(uint8_t addr)
{
  uint8_t BM22S2021_1_data=0;
  uint8_t BM22S2021_1_uniAck[8]; 
  uint16_t BM22S2021_1_num;    
  uint8_t BM22S2021_1_uniCmd[4] = {0xD0,0x00,0x00,0x00};
  BM22S2021_1_uniCmd[1] = addr;
  BM22S2021_1_num=0xD0 + addr +0x00;
  BM22S2021_1_uniCmd[3] = ~((uint8_t)(BM22S2021_1_num)) + 1;
  BM22S2021_1_uniCmd[1] = addr;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 8);
  _BM22S2021_1_delay(70);
  if (_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100) == 0x00 && BM22S2021_1_uniAck[4] == 0xD0)
  {
    BM22S2021_1_data=BM22S2021_1_uniAck[6];
  }
  _BM22S2021_1_delay(20);
  return  BM22S2021_1_data;
}


/*********************************************************************************************************//**
 * @brief   Read the run variable at the specified address
 * @param   addr:Run variable address
             All available addresses have been defined in the H file
             Run variable list:
                    @arg Equipment status :0X90
                    @arg VBG voltage a/d value(H+L) :0X91 0X92
                    @arg Reference value(H+L) :0X93 0X94
                    @arg T0A status (channel a) :0X95
                    @arg T0B status (channel B) :0X96
                    @arg T0B smoke detection value(H+L) :0X9b  0X9c
                    @arg T0B smoke detection value(H+L) :0X9D  0X9E
                    @arg T0A calibration zero(H+L) :0X9F 0XA0
                    @arg T0B calibration zero(H+L) :0XA1 0XA2
                    @arg T0A alarm threshold(H+L) :0XA3  0XA4
                    @arg T0B alarm threshold(H+L) :0XA5  0XA6
                    @arg A. Variation ratio of channel B :0XA7
                    @arg Alarm count :0XAA
                    @arg Temperature a/d value(H+L) :0XAB  0XAC
                    @arg Fault count :0XAF
                    @arg Smoke count :0XB0
                    @arg VDD voltage(H+L) :0XB1  0XB2 
 * @retval   data:Store read data
             0:read error
 ************************************************************************************************************/
uint8_t BM22S2021_1_readRunningVariables(uint8_t addr)
{
  uint8_t BM22S2021_1_data=0;  
  uint8_t BM22S2021_1_uniAck[8];
  uint16_t BM22S2021_1_num;  
  uint8_t BM22S2021_1_uniCmd[4] = {0xD2,0x00,0x00,0x00};
  BM22S2021_1_uniCmd[1] = addr;
  BM22S2021_1_num=0xD2 + addr + 0x00;
  BM22S2021_1_uniCmd[1] = addr;
  BM22S2021_1_uniCmd[3] = ~((uint8_t)(BM22S2021_1_num)) + 1;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  if (_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100) == 0x00 && BM22S2021_1_uniAck[4] == 0xD2)
  {
    BM22S2021_1_data=BM22S2021_1_uniAck[6];
  }
  _BM22S2021_1_delay(20);
  return  BM22S2021_1_data;
}


/*********************************************************************************************************//**
 * @brief   Get the current serial port automatic output status
 * @param   void 
 * @retval  0x80 Detailed data is automatically output for each smoke detection
            0x81 only Simple data output
            0x00 does not automatically output data;
 ************************************************************************************************************/
uint8_t BM22S2021_1_getAutoTx(void)
{
  uint8_t BM22S2021_1_mode=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_mode=BM22S2021_1_readRegister(0x2e);
  return BM22S2021_1_mode;
}


/*********************************************************************************************************//**
 * @brief   Get the level that the STATUS pin will output under the current alarm state
 * @param   void 
 * @retval  0x80 HIGH level
            0x00 LOW level
 ************************************************************************************************************/
uint8_t BM22S2021_1_getStatusPinActiveMode(void)
{
  uint8_t BM22S2021_1_level=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_level=BM22S2021_1_readRegister(0x2f);
  return BM22S2021_1_level;
}


/*********************************************************************************************************//**
 * @brief   get T0A calibration Top limits
 * @param   void 
 * @retval  toplimit:Air calibration upper limit
 ************************************************************************************************************/
uint8_t BM22S2021_1_getT0ATopLimit(void)
{
  uint8_t BM22S2021_1_toplimit=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_toplimit=BM22S2021_1_readRegister(0x08);
  return  BM22S2021_1_toplimit;
}


/*********************************************************************************************************//**
 * @brief   get T0A calibration Top limits
 * @param   void 
 * @retval  bottomlimit:Air calibration lower limit
 ************************************************************************************************************/
uint8_t BM22S2021_1_getT0ABottomLimit(void)
{
  uint8_t BM22S2021_1_bottomlimit=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_bottomlimit=BM22S2021_1_readRegister(0x09);
  return  BM22S2021_1_bottomlimit;
}


/*********************************************************************************************************//**
 * @brief   get T0B calibration Top limits
 * @param   void 
 * @retval  toplimit:Air calibration upper limit
 ************************************************************************************************************/
uint8_t BM22S2021_1_getT0BTopLimit(void)
{
  uint8_t BM22S2021_1_toplimit=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_toplimit=BM22S2021_1_readRegister(0x0A);
  return  BM22S2021_1_toplimit;
}


/*********************************************************************************************************//**
 * @brief   get T0B calibration Top limits
 * @param   void 
 * @retval  bottomlimit:Air calibration lower limit
 ************************************************************************************************************/
uint8_t BM22S2021_1_getT0BBottomLimit(void)
{
  uint8_t BM22S2021_1_gbottomlimit=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_gbottomlimit=BM22S2021_1_readRegister(0x0B);
  return  BM22S2021_1_gbottomlimit;
}


/*********************************************************************************************************//**
 * @brief   get the alarm threshold of T0A channel
 * @param   void 
 * @retval  Threshold:12bit valid data
 ************************************************************************************************************/
uint16_t BM22S2021_1_getT0AThreshold(void)
{
  uint16_t BM22S2021_1_Threshold=0;
  _BM22S2021_1_clear_UART_FIFO();  
 _BM22S2021_1_recBuf[0]=BM22S2021_1_readRegister(0x10);
  _BM22S2021_1_recBuf[1]=BM22S2021_1_readRegister(0x11);
  BM22S2021_1_Threshold=((uint16_t)_BM22S2021_1_recBuf[1]<<8 | _BM22S2021_1_recBuf[0]);
  return BM22S2021_1_Threshold;
}


/*********************************************************************************************************//**
 * @brief   get the alarm threshold of T0B channel
 * @param   void 
 * @retval  Threshold:12bit valid data
 ************************************************************************************************************/
uint16_t BM22S2021_1_getT0BThreshold(void)
{
  uint16_t BM22S2021_1_Threshold=0;
  _BM22S2021_1_clear_UART_FIFO();  
  _BM22S2021_1_recBuf[0]=BM22S2021_1_readRegister(0x12);
  _BM22S2021_1_recBuf[1]=BM22S2021_1_readRegister(0x13);
  BM22S2021_1_Threshold=(_BM22S2021_1_recBuf[1]<<8 | _BM22S2021_1_recBuf[0]);
  return BM22S2021_1_Threshold;
}


/*********************************************************************************************************//**
 * @brief   etection cycle of smoke-free environment equipment in standby, unit: s
 * @param   void 
 * @retval  cycle:time
 ************************************************************************************************************/
uint8_t BM22S2021_1_getDetectCycle(void)
{
  uint8_t BM22S2021_1_cycle=0;
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_cycle=BM22S2021_1_readRegister(0x2d);
  return BM22S2021_1_cycle;
}



/*********************************************************************************************************//**
 * @brief   Trigger the air calibration function. 
            The calibration time is 8s. The data bits in the calibration process will return to the current calibration timing in real time.
            The calibration is completed  The data bit returns 0xa0, and the failure returns 0xf0
 * @param   void 
 * @retval  0: module feedback is correct
            1: module feedback error
 ************************************************************************************************************/
uint8_t BM22S2021_1_calibrateModule(void)
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xAB, 0x00, 0x00, 0x55};
  uint8_t BM22S2021_1_uniAck[8];
  uint8_t BM22S2021_1_i;
  uint8_t BM22S2021_1_c;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  for(BM22S2021_1_c=0;BM22S2021_1_c<9;BM22S2021_1_c++)
  {
    BM22S2021_1_i=_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100);
    if(BM22S2021_1_c<8)
    {
      _BM22S2021_1_delay(1000);
    }
  }
  if (BM22S2021_1_i == 0x00 && BM22S2021_1_uniAck[6]==0xa0)
  {
    _BM22S2021_1_delay(20);  
    return    0;
  }
  else
  {
    _BM22S2021_1_delay(20);  
    return   1;
  }
}


/*********************************************************************************************************//**
 * @brief   Send instructions to reset the chip in the module
 * @param   void 
 * @retval   1: module data acquisition failed, there is no correct feedback value
             0: Module data obtained successfully
 ************************************************************************************************************/
uint8_t BM22S2021_1_resetModule(void)
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xAF, 0x00, 0x00, 0x51};
  uint8_t BM22S2021_1_uniAck[8];
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  if (_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100) == 0x00 && BM22S2021_1_uniAck[4] == 0xAF)
  {
    _BM22S2021_1_delay(20);
    return    0;
  }
  else
  {
    _BM22S2021_1_delay(20);
    return   1;
  }
}


/*********************************************************************************************************//**
 * @brief   Send command to restore the module to factory settings
 * @param   void 
 * @retval  1: Module setting failed without correct feedback value
            0: Module set successfully
 * @Others  Factory settings:
           @arg TOA and T0B
           @arg Calibration upper limit: 0xc8
           @arg Lower calibration limit: 0x19
           @arg T0A  Alarm threshold(L[7:0]:0x5e H[15:8]:0x01)
           @arg T0B  Alarm threshold(L[7:0]:0x96 H[15:8]:0x00)
           @arg Standby smoke detection cycle :  0x08
           @arg Serial port automatic output:  0x80
           @arg Alarm output level:0x80 HIGH
 ************************************************************************************************************/
uint8_t BM22S2021_1_restoreDefault(void)
{
  uint8_t BM22S2021_1_uniCmd[4] = {0xA0,0x00,0x00,0x60};
  uint8_t BM22S2021_1_uniAck[8];
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(350);
  if (_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100) == 0x00 && BM22S2021_1_uniAck[4] == 0xA0)
  {
   _BM22S2021_1_delay(20);
   return  0;
  }
  else
  {
    _BM22S2021_1_delay(20);    
    return  1;
  }
}


/*********************************************************************************************************//**
 * @brief   Write data to the register of the specified address
 * @param   addr:Destination register address to Write
            All available addresses have been defined in the H file
            Register list:
             @arg T0A upper / lower calibration limit (H+L):0X08  0X09
             @arg T0B upper / lower calibration limit (H+L):0X0A  0X0B
             @arg T0A alarm threshold(H+L):0X10 0X11
             @arg T0B alarm threshold(H+L):0X12 0X13
             @arg standby smoke detection cycle:0X2D
             @arg serial port automatic output:0X2E
             @arg alarm output level:0X2F
           data:Data to be written 
 * @retval  0: Write data successfully, module feedback is correct
            1: module feedback error
 ************************************************************************************************************/
uint8_t BM22S2021_1_writeRegister(uint8_t addr,uint8_t data)
{
  uint8_t BM22S2021_1_uniAck[8];
  uint8_t BM22S2021_1_uniCmd[4] = {0xE0, 0x00, 0x00,0x00};
  uint16_t BM22S2021_1_num=0xE0 + addr + data;
  BM22S2021_1_uniCmd[1] = addr;
  BM22S2021_1_uniCmd[2] = data;
  BM22S2021_1_uniCmd[3] = ~((uint8_t)BM22S2021_1_num) + 1;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_writeBytes(BM22S2021_1_uniCmd, 4);
  _BM22S2021_1_delay(50);
  if(addr==0x2e)
  {
    _BM22S2021_1_autoTxMode=data;
  }
if (_BM22S2021_1_readBytes(BM22S2021_1_uniAck,8,100) == 0x00 && BM22S2021_1_uniAck[4] == 0xE0)
  {
     _BM22S2021_1_delay(20);
     return    0;
  }
  else
  {
    _BM22S2021_1_delay(20);  
    return   1;
  }
}


/*********************************************************************************************************//**
 * @brief   Set the enable mode of TX pin automatic output
 * @param   mode:0x81-20byte 0x80--41byte  0x00--0byt 
 * @retval  0:Setting succeeded
            1:Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setAutoTx(uint8_t mode)
{
   uint8_t BM22S2021_1_sign;
   _BM22S2021_1_clear_UART_FIFO(); 
   BM22S2021_1_sign = BM22S2021_1_writeRegister(0x2e,mode);
   _BM22S2021_1_autoTxMode=mode;
   if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}


/*********************************************************************************************************//**
 * @brief   set the current serial port automatic output status
 * @param   state:0x80--HIGH level 0x0--LOW level
 * @retval  0:Setting succeeded
            1:Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setStatusPinActiveMode(uint8_t state)
{
    uint8_t BM22S2021_1_sign;
    _BM22S2021_1_clear_UART_FIFO();
    BM22S2021_1_sign = BM22S2021_1_writeRegister(0x2f,state);
    if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}



/*********************************************************************************************************//**
 * @brief   Set T0A calibration upper and lower limits
 * @param   toplimit:Air calibration upper limit
            bottomlimit:Air calibration lower limit
 * @retval  0: set successfully
            1: Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setT0ACalibrateRange(uint8_t toplimit,uint8_t bottomlimit)
{
   uint8_t BM22S2021_1_sign;  
  _BM22S2021_1_clear_UART_FIFO();
  BM22S2021_1_writeRegister(0x08,toplimit);
  BM22S2021_1_sign=BM22S2021_1_writeRegister(0x09,bottomlimit);
  if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}



/*********************************************************************************************************//**
 * @brief   Set T0B calibration upper and lower limits
 * @param   toplimit:Air calibration upper limit
            bottomlimit:Air calibration lower limit
 * @retval  0: set successfully
            1: Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setT0BCalibrateRange(uint8_t toplimit,uint8_t bottomlimit)
{
   uint8_t BM22S2021_1_sign;
   _BM22S2021_1_clear_UART_FIFO();
   BM22S2021_1_writeRegister(0x0A,toplimit);
   BM22S2021_1_sign=BM22S2021_1_writeRegister(0x0B,bottomlimit);
   if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}


/*********************************************************************************************************//**
 * @brief   set the alarm threshold of T0A channel
 * @param   value:16bit  default:0x015e
 * @retval  0: set successfully
            1: Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setT0AThreshold(uint16_t value)
{
  uint8_t BM22S2021_1_sign;
  _BM22S2021_1_recBuf[0]=(uint8_t)(value);
  _BM22S2021_1_recBuf[1]=(uint8_t)(value>>8);
  _BM22S2021_1_clear_UART_FIFO();  
  BM22S2021_1_writeRegister(0x10,_BM22S2021_1_recBuf[0]);
  BM22S2021_1_sign=BM22S2021_1_writeRegister(0x11,_BM22S2021_1_recBuf[1]);
  if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}


/*********************************************************************************************************//**
 * @brief   set the alarm threshold of T0B channel
 * @param   value:16bit  default:0x0096
 * @retval  0: set successfully
            1: Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setT0BThreshold(uint16_t value)
{
  uint8_t BM22S2021_1_sign;
  _BM22S2021_1_clear_UART_FIFO();
  _BM22S2021_1_recBuf[0]=(uint8_t)(value);
  _BM22S2021_1_recBuf[1]=(uint8_t)(value>>8);
   _BM22S2021_1_clear_UART_FIFO();
  BM22S2021_1_writeRegister(0x12,_BM22S2021_1_recBuf[0]);
  BM22S2021_1_sign = BM22S2021_1_writeRegister(0x13,_BM22S2021_1_recBuf[1]);
  if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}


/*********************************************************************************************************//**
 * @brief   set the alarm threshold of T0B channel
 * @param   value:16bit  default:0x0096
 * @retval  0: set successfully
            1: Setting failed
 ************************************************************************************************************/
uint8_t BM22S2021_1_setDetectCycle(uint8_t Cycle)
{
   uint8_t BM22S2021_1_sign;
   _BM22S2021_1_clear_UART_FIFO();
   BM22S2021_1_sign = BM22S2021_1_writeRegister(0x2d,Cycle);
    if(BM22S2021_1_sign==0)
  {
    return    0;
  }
  else
  {
    return    1;
  }
}

/*********************************************************************************************************//**
 * @brief   eliminate buff data
 * @param   void
 * @retval  void
 ************************************************************************************************************/
void _BM22S2021_1_clear_UART_FIFO(void)
{
  uint8_t BM22S2021_1_rbuf[1];  
   while (UARTM_GetReadBufferLength(gBM22S2021_1_SERIAL) > 0)
  {
    UARTM_ReadByte(gBM22S2021_1_SERIAL,BM22S2021_1_rbuf);
  }
}

/*********************************************************************************************************//**
 * @brief  write data through UART.
 * @param  wbuf:Variables for storing Data to be sent
           wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void _BM22S2021_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	 UARTM_DiscardReadBuffer(gBM22S2021_1_SERIAL);
	 UARTM_Write(gBM22S2021_1_SERIAL,wbuf,wlen);
}


/*********************************************************************************************************//**
 * @brief  read data through UART.
           rbuf:Variables for storing Data to be obtained
           rlen:Length of data to be obtained
           timOut:The timeout between reading the first byte and each byte,Unit (ms)
 * @retval BM22S2021_1_TIMEOUT_ERROR/BM22S2021_1_CHECK_OK/BM22S2021_1_CHECK_ERROR
 ************************************************************************************************************/
uint8_t _BM22S2021_1_readBytes(uint8_t rbuf[], uint8_t rlen, uint16_t timOut)
{
	  uint8_t _lenCount = 0;
	  uint8_t _delayCount = 0;
	  uint8_t _BM22S2021_1_checkSum = 0;
		 for(_lenCount = 0; _lenCount < rlen; _lenCount++)
		 {
			 _delayCount = 0;
			 while(UARTM_GetReadBufferLength(gBM22S2021_1_SERIAL) == 0)
			 {
				 if(_delayCount > timOut)
				 {
					 return BM22S2021_1_TIMEOUT_ERROR;
				 }
				 
				 _BM22S2021_1_delay(1);
				 _delayCount++;
			 }
			 UARTM_ReadByte(gBM22S2021_1_SERIAL,rbuf+_lenCount);		 
		 }

	 //checkSum
  for (_lenCount = 0; _lenCount < (rlen - 1); _lenCount++)
 {
    _BM22S2021_1_checkSum += rbuf[_lenCount];
  }
   _BM22S2021_1_checkSum = ~_BM22S2021_1_checkSum +1;
  if (_BM22S2021_1_checkSum == rbuf[rlen - 1])
  {
    return BM22S2021_1_CHECK_OK; // Check correct
  }
  else
 {
   return BM22S2021_1_CHECK_ERROR; // Check error
  }
}



/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count: Delay time,Unit(ms)
 * @retval void
 ************************************************************************************************************/
void _BM22S2021_1_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

/* Private functions ---------------------------------------------------------------------------------------*/
