/*********************************************************************************************************//**
 * @file    BM22S3221_1_HT32/src/BM22S3221_1.c
 * @version V1.0.1
 * @date    2025-06-16
 * @brief   The function of BM22S3221_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM22S3221_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM22S3221_1_SERIAL = BM22S3221_1_SERIAL;
uint8_t _BM22S3221_1_recBuf[32] = {0};
const uint8_t _BM22S3221_1_preheat = 1, _BM22S3221_1_calibration = 0, _BM22S3221_1_preheatTime = 120, _BM22S3221_1_calTime = 120;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief  module serial number select.
 * @param  serial_number: select serial number.
 * @retval BM22S3221_1_selStatus          
           @arg BM22S3221_1_SUCCESS
           @arg BM22S3221_1_FAILURE
 ************************************************************************************************************/
BM22S3221_1_selStatus BM22S3221_1_selSerial(uint32_t serial_number)
{
  if(CHECK_SERIALn(serial_number) == -1)
  {
    return BM22S3221_1_FAILURE;
  }

  gBM22S3221_1_SERIAL = serial_number;
  return BM22S3221_1_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using UART communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM22S3221_1_Init(void)
{
	{
	USART_InitTypeDef USART_InitStructure = {0};
  USART_InitStructure.USART_BaudRate = 9600;
  USART_InitStructure.USART_WordLength = USART_WORDLENGTH_8B;
  USART_InitStructure.USART_StopBits = USART_STOPBITS_1;
  USART_InitStructure.USART_Parity = USART_PARITY_NO;
  USART_InitStructure.USART_Mode = USART_MODE_NORMAL;
  UARTM_Init(gBM22S3221_1_SERIAL, &USART_InitStructure, BM22S3221_1_UART_TXTIMEOUT);
  }
		//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BM22S3221_1_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BM22S3221_1_INTPIN_GPIO_ID, BM22S3221_1_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BM22S3221_1_INTPIN_GPIO_PORT, BM22S3221_1_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BM22S3221_1_INTPIN_GPIO_PORT, BM22S3221_1_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BM22S3221_1_INTPIN_GPIO_PORT, BM22S3221_1_INTPIN_GPIO_PIN, ENABLE);
    
}

/*********************************************************************************************************//**
 * @brief  Wait for the module warm-up to complete(about 120 seconds)
 * @param  void
 * @retval void  
 ************************************************************************************************************/
void BM22S3221_1_preheatCountdown(void)
{ 
  _BM22S3221_1_countdown(_BM22S3221_1_preheat);
}

/*********************************************************************************************************//**
 * @brief  Get STA Pin Status
 * @param  void
 * @retval STA Pin Status:
                    @arg 0:STA Pin output low level 
                    @arg 1:STA Pin output high level 
 ************************************************************************************************************/
uint8_t BM22S3221_1_getSTATUS(void)
{
	return (GPIO_ReadInBit(BM22S3221_1_INTPIN_GPIO_PORT,BM22S3221_1_INTPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  Get the current status of the module
 * @param  void
 * @retval module status (refer to datasheet for meaning of each bit)
 ************************************************************************************************************/
uint8_t BM22S3221_1_getWorkStatus(void)
{
  uint8_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf[4] = {0xD2, 0x88, 0x00, 0xA6};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
  }
  return BM22S3221_1_tmp;
}

/*********************************************************************************************************//**
 * @brief  Read CO concentration value
 * @param  void
 * @retval CO concentration value in ppm
 ************************************************************************************************************/
uint16_t BM22S3221_1_readCOValue(void)
{
  uint16_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf1[4] = {0xD2, 0x84, 0x00, 0xAA};
  uint8_t BM22S3221_1_sendBuf0[4] = {0xD2, 0x85, 0x00, 0xA9};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf1, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
    BM22S3221_1_tmp = BM22S3221_1_tmp << 8;
    _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf0, 4);
    _BM22S3221_1_delay(20); // Wait for the module to reply data
    if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
    {
      BM22S3221_1_tmp += BM22S3221_1_recBuf[6];
    }
    else
    {
      BM22S3221_1_tmp = 0;
    }
  }
  return BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Read CO AD value
 * @param  void
 * @retval AD value for CO concentration
 ************************************************************************************************************/
uint16_t BM22S3221_1_readADValue(void)
{
  uint16_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf1[4] = {0xD2, 0x80, 0x00, 0xAE};
  uint8_t BM22S3221_1_sendBuf0[4] = {0xD2, 0x81, 0x00, 0xAD};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf1, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
    BM22S3221_1_tmp = BM22S3221_1_tmp << 8;
    _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf0, 4);
    _BM22S3221_1_delay(20); // Wait for the module to reply data
    if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
    {
      BM22S3221_1_tmp += BM22S3221_1_recBuf[6];
    }
    else
    {
      BM22S3221_1_tmp = 0;
    }
  }
  return BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Read power-on reference value
 * @param  void
 * @retval power-on reference value
 ************************************************************************************************************/
uint16_t BM22S3221_1_readRefValue(void)
{
  uint16_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf1[4] = {0xD2, 0x82, 0x00, 0xAC};
  uint8_t BM22S3221_1_sendBuf0[4] = {0xD2, 0x83, 0x00, 0xAB};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf1, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
    BM22S3221_1_tmp = BM22S3221_1_tmp << 8;
    _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf0, 4);
    _BM22S3221_1_delay(20); // Wait for the module to reply data
    if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
    {
      BM22S3221_1_tmp += BM22S3221_1_recBuf[6];
    }
    else
    {
      BM22S3221_1_tmp = 0;
    }
  }
  return BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Get the current status and data of the module
 * @param  array: The array for storing the 32-byte module information
                  (refer to datasheet for meaning of each bit)
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 ************************************************************************************************************/
uint8_t BM22S3221_1_requestInfoPackage(uint8_t array[])
{
  uint8_t BM22S3221_1_i = 0;
  uint8_t BM22S3221_1_errFlag = 1;
  uint8_t BM22S3221_1_sendBuf[4] = {0xAC, 0x00, 0x00, 0x54};
  uint8_t BM22S3221_1_recBuf[32];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(40); // Wait for the module to reply data
  BM22S3221_1_errFlag = _BM22S3221_1_readBytes(BM22S3221_1_recBuf,32,20);
  if (BM22S3221_1_errFlag == 0)
  {
    for (BM22S3221_1_i = 0; BM22S3221_1_i < 32; BM22S3221_1_i++)
    {
      array[BM22S3221_1_i] = BM22S3221_1_recBuf[BM22S3221_1_i];
    }
  }
  return BM22S3221_1_errFlag;
}


/*********************************************************************************************************//**
 * @brief  Query whether the 32-byte data sent by the module is received
 * @param  void
 * @retval Whether the 32-byte data sent by the module has been received:
                    @arg TRUE(1): 32-byte data received 
                    @arg FALSE(0): 32-byte data not received
* @Others  Only used in the mode of Tx Auto Output Info
 ************************************************************************************************************/
bool BM22S3221_1_isInfoAvailable(void)
{
  uint8_t BM22S3221_1_header[5] = {0xAA, 0x20, 0x21, 0x01, 0xAC}; // Fixed code for first 5 bytes of 32-byte data
  uint8_t BM22S3221_1_recBuf[32] = {0}, BM22S3221_1_recLen = 32;
  uint8_t BM22S3221_1_i, BM22S3221_1_num = 0, BM22S3221_1_readCnt = 0, BM22S3221_1_failCnt = 0, BM22S3221_1_checkCode = 0;
  bool BM22S3221_1_isHeader = false, BM22S3221_1_result = false;
  uint8_t BM22S3221_1_rbuf[1];
  BM22S3221_1_num = UARTM_GetReadBufferLength(gBM22S3221_1_SERIAL);
  /* Serial buffer contains at least one 32-byte data */
  if (BM22S3221_1_num >= BM22S3221_1_recLen)
  {
    while (BM22S3221_1_failCnt < 2) // Didn't read the required data twice, exiting the loop
    {
      /* Find 5-byte data header */
      for (BM22S3221_1_i = 0; BM22S3221_1_i < 5;)
      {
        UARTM_ReadByte(gBM22S3221_1_SERIAL,BM22S3221_1_rbuf);
        BM22S3221_1_recBuf[BM22S3221_1_i] = BM22S3221_1_rbuf[0];
        if (BM22S3221_1_recBuf[BM22S3221_1_i] == BM22S3221_1_header[BM22S3221_1_i])
        {
          BM22S3221_1_isHeader = true; // Fixed code is correct
          BM22S3221_1_i++;             // Next byte
        }
        else if (BM22S3221_1_recBuf[BM22S3221_1_i] != BM22S3221_1_header[BM22S3221_1_i] && BM22S3221_1_i > 0)
        {
          BM22S3221_1_isHeader = false; // Next fixed code error
          BM22S3221_1_failCnt++;
          break;
        }
        else if (BM22S3221_1_recBuf[BM22S3221_1_i] != BM22S3221_1_header[BM22S3221_1_i] && BM22S3221_1_i == 0)
        {
          BM22S3221_1_readCnt++; // 0xAA not found, continue
        }
        if (BM22S3221_1_readCnt > (BM22S3221_1_num - BM22S3221_1_recLen))
        {
          return false; // No data found
        }
      }

      /* Find the correct fixed code */
      if (BM22S3221_1_isHeader)
      {
        for (BM22S3221_1_i = 0; BM22S3221_1_i < 5; BM22S3221_1_i++)
        {
          BM22S3221_1_checkCode += BM22S3221_1_recBuf[BM22S3221_1_i]; // Sum checkCode
        }
        for (BM22S3221_1_i = 5; BM22S3221_1_i < BM22S3221_1_recLen; BM22S3221_1_i++) // Read subsequent 27-byte data
        {
          UARTM_ReadByte(gBM22S3221_1_SERIAL,BM22S3221_1_rbuf);
          BM22S3221_1_recBuf[BM22S3221_1_i] = BM22S3221_1_rbuf[0];
          BM22S3221_1_checkCode += BM22S3221_1_recBuf[BM22S3221_1_i]; // Sum checkCode
        }
        BM22S3221_1_checkCode = BM22S3221_1_checkCode - BM22S3221_1_recBuf[BM22S3221_1_recLen - 1];
        BM22S3221_1_checkCode = (~BM22S3221_1_checkCode) + 1; // Calculate checkCode

        /* Compare whether the check code is correct */
        if (BM22S3221_1_checkCode == BM22S3221_1_recBuf[BM22S3221_1_recLen - 1])
        {
          for (BM22S3221_1_i = 0; BM22S3221_1_i < BM22S3221_1_recLen; BM22S3221_1_i++)
          {
            _BM22S3221_1_recBuf[BM22S3221_1_i] = BM22S3221_1_recBuf[BM22S3221_1_i]; // True, assign data to _recBuf[]
          }
          BM22S3221_1_result = true;
          break; // Exit "while (failCnt < 2)" loop
        }
        else
        {
          BM22S3221_1_failCnt++; // Error, failCnt plus 1, return "while (failCnt < 2)" loop
          BM22S3221_1_checkCode = 0;
        }
      }
    }
  }
  return BM22S3221_1_result;
}

/*********************************************************************************************************//**
 * @brief  Read the 32-byte data of sent by the module
 * @param  array: The array for storing the 32-byte module information
                  (refer to datasheet for meaning of each bit)
 * @retval void  
 * @Others Use after isInfoAvailable()
 ************************************************************************************************************/
void BM22S3221_1_readInfoPackage(uint8_t array[])
{
  uint8_t BM22S3221_1_i;
  for (BM22S3221_1_i = 0; BM22S3221_1_i < 32; BM22S3221_1_i++)
  {
    array[BM22S3221_1_i] = _BM22S3221_1_recBuf[BM22S3221_1_i];
  }
}


/*********************************************************************************************************//**
 * @brief  reset module
 * @param  void
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error 
 * @Others After the command is executed, the module needs to be preheated again
 ************************************************************************************************************/
uint8_t BM22S3221_1_resetModule(void)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xAF, 0x00, 0x00, 0x51}; // Used to store CMD to be sent
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20);
}


/*********************************************************************************************************//**
 * @brief  Restore module parameters to factory default values
 * @param  void
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error 
 * @Others  After the command is executed, the module needs to be preheated again.
            Factory set data description:
                    @arg Alarm threshold setting: 180ppm
                    @arg Exit alarm threshold setting: 55ppm
                    @arg Calibration zero point
                    @arg Data of the calibration with CO
                    @arg Restore factory setting
 ************************************************************************************************************/
uint8_t BM22S3221_1_restoreDefault(void)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xA0, 0x00, 0x00, 0x60};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(50); // Waiting for module to receive data and reply
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20);
}


/*********************************************************************************************************//**
 * @brief  Get firmware version
 * @param  void
 * @retval FW version, for example 0x0106: V1.6
 ************************************************************************************************************/
uint16_t BM22S3221_1_getFWVer(void)
{
  uint16_t BM22S3221_1_ver = 0;
  uint8_t BM22S3221_1_sendBuf[4] = {0xAD, 0x00, 0x00, 0x53};
  uint8_t BM22S3221_1_recBuf[12];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(30); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf, 12,20) == 0)
  {
    BM22S3221_1_ver = (BM22S3221_1_recBuf[6] << 8) + BM22S3221_1_recBuf[7];
  }
  return BM22S3221_1_ver;
}


/*********************************************************************************************************//**
 * @brief  Get production date
 * @param   array[0]: year
            array[1]: month
            array[2]: day
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 * @Others for example 0x22, 0x05, 0x06: 2022/5/6
 ************************************************************************************************************/
uint8_t BM22S3221_1_getProDate(uint8_t array[])
{
  uint8_t BM22S3221_1_i = 0;
  uint8_t BM22S3221_1_errFlag;
  uint8_t BM22S3221_1_sendBuf[4] = {0xAD, 0x00, 0x00, 0x53};
  uint8_t BM22S3221_1_recBuf[12];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(30); // Wait for the module to reply data
  BM22S3221_1_errFlag =_BM22S3221_1_readBytes(BM22S3221_1_recBuf,12,20);
  if (BM22S3221_1_errFlag == 0)
  {
    for (BM22S3221_1_i = 8; BM22S3221_1_i < 11; BM22S3221_1_i++)
    {
      array[BM22S3221_1_i - 8] = BM22S3221_1_recBuf[BM22S3221_1_i];
    }
  }
  return BM22S3221_1_errFlag;
}


/*********************************************************************************************************//**
 * @brief  Query whether the automatic serial output of the module is enabled
 * @param  void
 * @retval Whether the automatic output of the module serial port is enabled:
                    @arg TRUE(1): auto-Tx ENABLED
                    @arg false(0):auto-Tx DISENABLED
 ************************************************************************************************************/
bool BM22S3221_1_isAutoTx(void)
{
  uint8_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf[4] = {0xD0, 0x1B, 0x00, 0x15};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
  }
  if (BM22S3221_1_tmp == 0x08)
  {
    return true;
  }
  else
  {
    return false;
  }
}


/*********************************************************************************************************//**
 * @brief  Get the alarm output level of the STATUS pin
 * @param  void
 * @retval The alarm output level of the STATUS pin:
                    @arg 08 : STATUS pin is high when alarming, low when not alarming
                    @arg 00 : STATUS pin is low when alarming, high when not alarming
 ************************************************************************************************************/
uint8_t BM22S3221_1_getStatusPinActiveMode(void)
{
  uint8_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf[4] = {0xD0, 0x1C, 0x00, 0x14};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
  }
  return BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Get the alarm threshold
 * @param  void
 * @retval the threshold value for generating an alarm, in ppm. Default 180 ppm
 ************************************************************************************************************/
uint16_t BM22S3221_1_getAlarmThreshold(void)
{
  uint16_t BM22S3221_1_tmp = 0;
  uint8_t BM22S3221_1_sendBuf1[4] = {0xD0, 0x0C, 0x00, 0x24};
  uint8_t BM22S3221_1_sendBuf0[4] = {0xD0, 0x0D, 0x00, 0x23};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf1, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
  {
    BM22S3221_1_tmp = BM22S3221_1_recBuf[6];
    BM22S3221_1_tmp = BM22S3221_1_tmp << 8;
    _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf0, 4);
    _BM22S3221_1_delay(20); // Wait for the module to reply data
    if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20) == 0)
    {
      BM22S3221_1_tmp += BM22S3221_1_recBuf[6];
    }
    else
    {
      BM22S3221_1_tmp = 0;
    }
  }
  return BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Get the exit alarm threshold
 * @param  void
 * @retval the value to stop alarming, in ppm.When CO concentration is lower than this value, the alarm stops. Default 55 ppm
 ************************************************************************************************************/
uint8_t BM22S3221_1_getExitAlarmThreshold(void)
{
  uint8_t  BM22S3221_1_tmp = 0;
  uint8_t  BM22S3221_1_sendBuf[4] = {0xD0, 0x0E, 0x00, 0x22};
  uint8_t  BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes( BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  if (_BM22S3221_1_readBytes( BM22S3221_1_recBuf,8,20) == 0)
  {
     BM22S3221_1_tmp =  BM22S3221_1_recBuf[6];
  }
  return  BM22S3221_1_tmp;
}


/*********************************************************************************************************//**
 * @brief  Set whether Tx pin output data automatically
 * @param  autoTx = AUTO(08):enable to transmit the 34-byte module information on Tx pin automatically per second
           autoTx = PASSIVE(00): disable to transmit automatically
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 ************************************************************************************************************/
uint8_t BM22S3221_1_setAutoTx(uint8_t autoTx)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xE0, 0x1B, 0, 0};
  uint8_t BM22S3221_1_recBuf[8];
  BM22S3221_1_sendBuf[2] = autoTx;
  BM22S3221_1_sendBuf[3] = ~(BM22S3221_1_sendBuf[0] + BM22S3221_1_sendBuf[1] + BM22S3221_1_sendBuf[2]) + 1;
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20);
}


/*********************************************************************************************************//**
 * @brief  Set the alarm output level of the STATUS pin
 * @param  statusMode = HIGH_LEVEL(08): Status pin is high when alarming, low when not alarming
            statusMode = LOW_LEVEL(00): Status pin is low when alarming, high when not alarming
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 ************************************************************************************************************/
uint8_t BM22S3221_1_setStatusPinActiveMode(uint8_t statusMode)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xE0, 0x1C, 0, 0};
  uint8_t BM22S3221_1_recBuf[8];
  BM22S3221_1_sendBuf[2] = statusMode;
  BM22S3221_1_sendBuf[3] = ~(BM22S3221_1_sendBuf[0] + BM22S3221_1_sendBuf[1] + BM22S3221_1_sendBuf[2]) + 1;
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf, 8,20);
}


/*********************************************************************************************************//**
 * @brief  Set the threshold for module alarms
 * @param  alarmThreshold, in ppm. Default 180 ppm
            When CO concentration is higher than this value, alarm is set.
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 ************************************************************************************************************/
uint8_t BM22S3221_1_setAlarmThreshold(uint16_t alarmThreshold)
{
  uint8_t BM22S3221_1_sendBuf1[4] = {0xE0, 0x0C, 0, 0};
  uint8_t BM22S3221_1_sendBuf0[4] = {0xE0, 0x0D, 0, 0};
  uint8_t BM22S3221_1_recBuf[8], BM22S3221_1_errFlag;
  BM22S3221_1_sendBuf1[2] = alarmThreshold >> 8;
  BM22S3221_1_sendBuf0[2] = alarmThreshold & 0xFF;
  BM22S3221_1_sendBuf1[3] = ~(BM22S3221_1_sendBuf1[0] + BM22S3221_1_sendBuf1[1] + BM22S3221_1_sendBuf1[2]) + 1;
  BM22S3221_1_sendBuf0[3] = ~(BM22S3221_1_sendBuf0[0] + BM22S3221_1_sendBuf0[1] + BM22S3221_1_sendBuf0[2]) + 1;
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf1, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  BM22S3221_1_errFlag = _BM22S3221_1_readBytes(BM22S3221_1_recBuf, 8,20);
  if (BM22S3221_1_errFlag == 0)
  {
    _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf0, 4);
    _BM22S3221_1_delay(20); // Wait for the module to reply data
    return _BM22S3221_1_readBytes(BM22S3221_1_recBuf, 8,20);
  }
  else
  {
    return BM22S3221_1_errFlag;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the threshold for module stop alarm
 * @param  exitAlarmThreshold, in ppm. Default 55 ppm
            When CO concentration is lower than this value, stop alarming.
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 ************************************************************************************************************/
uint8_t BM22S3221_1_setExitAlarmThreshold(uint8_t exitAlarmThreshold)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xE0, 0x0E, 0, 0};
  uint8_t BM22S3221_1_recBuf[8];
  BM22S3221_1_sendBuf[2] = exitAlarmThreshold;
  BM22S3221_1_sendBuf[3] = ~(BM22S3221_1_sendBuf[0] + BM22S3221_1_sendBuf[1] + BM22S3221_1_sendBuf[2]) + 1;
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf, 8,20);
}


/*********************************************************************************************************//**
 * @brief  Calibrate module
 * @param  void
 * @retval Communication situation:
                    @arg 0:Request ok 
                    @arg 1:Check error
                    @arg 2: Timeout error
 * @Others 
           @arg Trigger the calibration with gas,which needs to be carried out in the standard concentration gas environment,(For example, in the gas concentration environment of 10% LEL / 5000ppm)
           @arg The module enters the calibration state after receiving this command.Do not send other commands to the module during this period
 ************************************************************************************************************/
uint8_t BM22S3221_1_calibrateModule(void)
{
  uint8_t BM22S3221_1_sendBuf[4] = {0xAB, 0xB1, 0x00, 0xA4};
  uint8_t BM22S3221_1_recBuf[8];
  _BM22S3221_1_writeBytes(BM22S3221_1_sendBuf, 4);
  _BM22S3221_1_delay(20); // Wait for the module to reply data
  return _BM22S3221_1_readBytes(BM22S3221_1_recBuf,8,20);
}


/*********************************************************************************************************//**
 * @brief  Wait for the module calibration to complete(about 120 seconds)
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM22S3221_1_calibrateCountdown(void)
{
  _BM22S3221_1_countdown(_BM22S3221_1_calibration);
}


/*********************************************************************************************************//**
 * @brief  Clear UART Receive FIFO
 * @param  void
 * @retval void
 ************************************************************************************************************/
void  _BM22S3221_1_clear_UART_FIFO(void)
{
  uint8_t rbuf[1];  
   while (UARTM_GetReadBufferLength(gBM22S3221_1_SERIAL) > 0)
  {
    UARTM_ReadByte(gBM22S3221_1_SERIAL,rbuf);
  }
}

/*********************************************************************************************************//**
 * @brief  Countdown
 * @param  typeSelect countdown
                    @arg 1: Perform preheat countdown
                    @arg 0: Perform calibration countdown
 * @retval void
 ************************************************************************************************************/
void  _BM22S3221_1_countdown(uint8_t type)
{
  uint8_t BM22S3221_1_countdown, BM22S3221_1_tmp = 0, BM22S3221_1_i;
  uint8_t BM22S3221_1_delayTime = 120, BM22S3221_1_num = 14, BM22S3221_1_txMode = BM22S3221_1_AUTO;
  uint8_t BM22S3221_1_recBuf[32] = {0}, BM22S3221_1_recLen = 32;
  uint16_t BM22S3221_1_delayCnt = 0;

  if (type == _BM22S3221_1_preheat)
  {
    BM22S3221_1_delayTime = _BM22S3221_1_preheatTime;
    BM22S3221_1_num = 14;
  }
  else if (type == _BM22S3221_1_calibration)
  {
    BM22S3221_1_delayTime = _BM22S3221_1_calTime;
    BM22S3221_1_num = 15;
  }
  do
  {
    BM22S3221_1_delayCnt = 0;
    BM22S3221_1_tmp = 0;
    _BM22S3221_1_clear_UART_FIFO();
    while (BM22S3221_1_tmp < BM22S3221_1_recLen)
    {
      BM22S3221_1_tmp = UARTM_GetReadBufferLength(gBM22S3221_1_SERIAL);
      _BM22S3221_1_delay(1);
      BM22S3221_1_delayCnt++;
      if (BM22S3221_1_delayCnt > 3000)
      {
        BM22S3221_1_txMode = BM22S3221_1_PASSIVE; // 32-byte of data are not received within 3 seconds
        break;
      }
    }
    if (BM22S3221_1_txMode == BM22S3221_1_PASSIVE)
    {
      break; // Exit the do {...} While loop
    }
    if (_BM22S3221_1_readBytes(BM22S3221_1_recBuf, BM22S3221_1_recLen,20) == 0)
    {
      BM22S3221_1_countdown = BM22S3221_1_recBuf[BM22S3221_1_num];
    }
  } while (BM22S3221_1_countdown != 0);

  /* auto-Tx DISENABLED, delay "delayTime" sec*/
  if (BM22S3221_1_txMode == BM22S3221_1_PASSIVE)
  {
    for (BM22S3221_1_i = BM22S3221_1_delayTime; BM22S3221_1_i > 0; BM22S3221_1_i--)
    {
      _BM22S3221_1_delay(1030);
    }
  }
}


/*********************************************************************************************************//**
 * @brief  write data through UART.
 * @param  wbuf:Variables for storing Data to be sent
           wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void _BM22S3221_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	 UARTM_DiscardReadBuffer(gBM22S3221_1_SERIAL);
	 UARTM_Write(gBM22S3221_1_SERIAL,wbuf,wlen);
}

/*********************************************************************************************************//**
 * @brief  read data through UART.
           rbuf:Variables for storing Data to be obtained
           rlen:Length of data to be obtained
           timOut:The timeout between reading the first byte and each byte,Unit (ms)
 * @retval BM22S3221_1_TIMEOUT_ERROR/BM22S3221_1_CHECK_OK/BM22S3221_1_CHECK_ERROR
 ************************************************************************************************************/
uint8_t _BM22S3221_1_readBytes(uint8_t rbuf[], uint8_t rlen, uint16_t timOut)
{
	  uint8_t _lenCount = 0;
	  uint8_t _delayCount = 0;
	  uint8_t _BM22S3221_1_checkSum = 0;
		 for(_lenCount = 0; _lenCount < rlen; _lenCount++)
		 {
			 _delayCount = 0;
			 while(UARTM_GetReadBufferLength(gBM22S3221_1_SERIAL) == 0)
			 {
				 if(_delayCount > timOut)
				 {
					 return BM22S3221_1_TIMEOUT_ERROR;
				 }
				 
				 _BM22S3221_1_delay(1);
				 _delayCount++;
			 }
			 UARTM_ReadByte(gBM22S3221_1_SERIAL,rbuf+_lenCount);		 
		 }

	 //checkSum
  for (_lenCount = 0; _lenCount < (rlen - 1); _lenCount++)
 {
    _BM22S3221_1_checkSum += rbuf[_lenCount];
  }
   _BM22S3221_1_checkSum = ~_BM22S3221_1_checkSum +1;
  if (_BM22S3221_1_checkSum == rbuf[rlen - 1])
  {
    return BM22S3221_1_CHECK_OK; // Check correct
  }
  else
 {
   return BM22S3221_1_CHECK_ERROR; // Check error
  }
}



/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count: Delay time,Unit(ms)
 * @retval void
 ************************************************************************************************************/
void _BM22S3221_1_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

/* Private functions ---------------------------------------------------------------------------------------*/
