/*************************************************************************************************************
 * @file    BM25S2621_1_HT32/src/BM25S2621_1.c
 * @version V1.0.1
 * @date    2025-06-20
 * @brief   The function of BM25S2621_1 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BM25S2621_1.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
#if (CHECK_SERIALn(BM25S2621_1_SERIAL) == -1)
#error "Error: BM25S2621_1_SERIAL doesn't support the current SERIAL number. Check ht32_board_config.h."
#endif
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBM25S2621_1_SERIAL = BM25S2621_1_SERIAL;
/* Private variables ---------------------------------------------------------------------------------------*/
uint16_t _gRangeMax = 2000;
vu32 _gBFTM0Count; 
/* Global functions ----------------------------------------------------------------------------------------*/
/************************************************************************************************************
 * @brief module serial part number select.
 * @param serial_number: select serial number.
 * @retval BM25S2621_1_SUCCESS/BM25S2621_1_FAILURE
 ************************************************************************************************************/
BM25S2621_1_selStatus BM25S2621_1_selSerial(uint32_t serial_number)
{
  if (CHECK_SERIALn(serial_number) == -1)
  {
    return BM25S2621_1_FAILURE;
  }
  gBM25S2621_1_SERIAL = serial_number;
  return BM25S2621_1_SUCCESS;
}

/***********************************************************************************************************
 * @brief  Module initialization using UART communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BM25S2621_1_Init(void)
{
  USART_InitTypeDef USART_InitStructure = {0};
  USART_InitStructure.USART_BaudRate = 9600;
  USART_InitStructure.USART_WordLength = USART_WORDLENGTH_8B;
  USART_InitStructure.USART_StopBits = USART_STOPBITS_1;
  USART_InitStructure.USART_Parity = USART_PARITY_NO;
  USART_InitStructure.USART_Mode = USART_MODE_NORMAL;
  UARTM_Init(gBM25S2621_1_SERIAL, &USART_InitStructure, BM25S2621_1_UART_TXTIMEOUT);
  // D22(STATUS1):PWM capture
  { /* Enable peripheral clock                                                                              */
    CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
    CKCUClock.Bit.BFTM0 = 1;
    CKCU_PeripClockConfig(CKCUClock, ENABLE);
  }

  /* BFTM as Repetitive mode to trigger the match interrupt                                                 */
  BFTM_SetCompare(HT_BFTM0, SystemCoreClock / 100000); // 10us
  BFTM_SetCounter(HT_BFTM0, 0);
  BFTM_IntConfig(HT_BFTM0, ENABLE);
  BFTM_EnaCmd(HT_BFTM0, ENABLE);
  NVIC_DisableIRQ(BFTM0_IRQn);
	
  // INPUT:D22<--->STA Pin(PWM)
  { /* Enable peripheral clock                                                                              */
    CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
    CKCUClock.Bit.BM25S2621_1_STA_PIN_GPIO = 1;
    CKCUClock.Bit.AFIO = 1;
    CKCU_PeripClockConfig(CKCUClock, ENABLE);
  }
  /* Configure AFIO mode as GPIO                                                                          */
  AFIO_GPxConfig(BM25S2621_1_STA_PIN_GPIO_ID, BM25S2621_1_STA_PIN_AFIO_PIN, AFIO_FUN_GPIO);
  /* Configure GPIO direction as input                                                                    */
  GPIO_DirectionConfig(BM25S2621_1_STA_PIN_GPIO_PORT, BM25S2621_1_STA_PIN_GPIO_PIN, GPIO_DIR_OUT);
}
/***********************************************************************************************************
 * @brief  set module's ID(ID range: 1~254)
 * @param  currentID: current ID of module(default : 1)
 * @param  newID: new ID of module
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval result
 *    @arg 0: Susses
 *    @arg 1: Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_setID(uint8_t currentID, uint8_t newID, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = currentID;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_WRITE, BM25S2621_1_ADDR_ID, newID, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}

/***********************************************************************************************************
 * @brief  get module's ID
 * @param  void
 * @retval result
 *    @arg 1~254: module's ID
 *    @arg 0:communication failure
 ************************************************************************************************************/
uint8_t BM25S2621_1_getID(void)
{
  uint8_t readbuffer[9];
  if (_BM25S2621_1_RS485_request(0xff, BM25S2621_1_FUNCODE_READ, BM25S2621_1_ADDR_ID, 1, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    return readbuffer[4];
  }
  else
  {
    return BM25S2621_1_CHECK_ERROR;
  }
}

/*********************************************************************************************************
 * @brief  Read module's temperature measurement value
 * @param  id: current ID of module
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval Module's temperature measurement value(unit: celsius)
 **********************************************************************************************************/
uint8_t BM25S2621_1_readTemperature(uint8_t id, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_READ, BM25S2621_1_ADDR_TEMP, 1, readbuffer) == BM25S2621_1_CHECK_OK)
    return (readbuffer[3] << 8) | readbuffer[4];
  else
    return BM25S2621_1_CHECK_ERROR;
}

/*********************************************************************************************************
 * @brief  Read module's moisture measurement value
 * @param  id: current ID of module
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval Module's moisture measurement value(unit: %RH)
 **********************************************************************************************************/
uint8_t BM25S2621_1_readMoisture(uint8_t id, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_READ, BM25S2621_1_ADDR_MOIS, 1, readbuffer) == BM25S2621_1_CHECK_OK)
    return (readbuffer[3] << 8) | readbuffer[4];
  else
    return BM25S2621_1_CHECK_ERROR;
}

/***********************************************************************************************************
 * @brief  Read module's moisture and temperature measurement value
 * @param  id: current ID of module
 * @param  *temp: Address for storing module's temperature measurement value(unit: celsius)
 * @param  *mois: Address for storing module's moisture measurement value(unit: %RH)
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_readTemperatureAndMoisture(uint8_t id, uint8_t *temp, uint8_t *mois, bool broadcast)
{
  uint8_t readbuffer[15];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_READ, BM25S2621_1_ADDR_MOIS, 2, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    *mois = (readbuffer[3] << 8) | readbuffer[4];
    *temp = (readbuffer[5] << 8) | readbuffer[6];
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}
/***********************************************************************************************************
 * @brief   Restore default
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_restoreDefault(bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = 1;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_WRITE, BM25S2621_1_ADDR_ID, 1, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    _BM25S2621_1_delayMs(5);
    if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_WRITE, BM25S2621_1_ADDR_TEMPCALI, 0x0000, readbuffer) == BM25S2621_1_CHECK_OK)
    {
      _BM25S2621_1_delayMs(5);
      if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_WRITE, BM25S2621_1_ADDR_MOISCALI, 0x0000, readbuffer) == BM25S2621_1_CHECK_OK)
      {
        return BM25S2621_1_CHECK_ERROR;
      }
      else
        return BM25S2621_1_CHECK_OK;
    }
    else
      return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_OK;
}
/*********************************************************************************************************
 * @brief  send specified commands to the module
 * @param  *ptr_command: store the array address of the command
 * @param  *ptr_ack: The array address of the storage module's response
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 **********************************************************************************************************/
uint8_t BM25S2621_1_sendCommand(uint8_t *ptr_command, uint8_t *ptr_ack)
{
  uint8_t send_id = ptr_command[0];
  uint8_t send_funcode = ptr_command[1];
  uint16_t send_start_addr = (ptr_command[2] << 8) | ptr_command[3];
  uint16_t send_len = (ptr_command[4] << 8) | ptr_command[5];
  if (_BM25S2621_1_RS485_request(send_id, send_funcode, send_start_addr, send_len, ptr_ack) == BM25S2621_1_CHECK_OK)
  {
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}

/***********************************************************************************************************
 * @brief  calibrate module's temperature
 * @param  id: current ID of module
 * @param  currentTemp: current soil temperature
 * @param  broadcast: choose whether it is a broadcast command(default : false)
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_calibrateTemperature(uint8_t id, uint8_t currentTemp, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_TEMPCALI, 0, currentTemp, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}

/***********************************************************************************************************
 * @brief  calibrate module's moisture
 * @param  id : current ID of module
 * @param  currentMois : current soil moisture
 * @param  broadcast : choose whether it is a broadcast command(default : false)
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_calibrateMoisture(uint8_t id, uint8_t currentMois, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_FUNCODE_MOISCALI, 0, currentMois, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}

/***********************************************************************************************************
 * @brief  empty water calibration
 * @param  id : current ID of module
 * @param  broadcast : choose whether it is a broadcast command(default : false)
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t BM25S2621_1_emptyWaterCalibrate(uint8_t id, bool broadcast)
{
  uint8_t readbuffer[9];
  uint8_t module_id;
  if (broadcast)
    module_id = 0xff;
  else
    module_id = id;
  if (_BM25S2621_1_RS485_request(module_id, BM25S2621_1_ADDR_MOISCALI, 0, 0, readbuffer) == BM25S2621_1_CHECK_OK)
  {
    return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}
/***********************************************************************************************************
 * @brief  Modbus CRC caculate
 * @param  ptr : data
 * @param  len : data length
 * @param  crc_H:crc Hight
 * @param  crc_L:crc Low
 * @retval void
 ************************************************************************************************************/
void _BM25S2621_1_Modbus_CRCcaculate(unsigned char *ptr, int len, u8 *crc_H, u8 *crc_L)
{
  unsigned int i;
  unsigned short crc = 0xFFFF;

  while (len--)
  {
    crc ^= *ptr++;
    for (i = 0; i < 8; ++i)
    {
      if (crc & 1)
        crc = (crc >> 1) ^ 0xA001;
      else
        crc = (crc >> 1);
    }
  }
  *crc_L = crc & 0x00ff;
  *crc_H = (crc >> 8) & 0x00ff;
}
/***********************************************************************************************************
 * @brief  RS485 request
 * @param  id : current ID of module
 * @param  funcode : choose whether it is a broadcast command(default : false)
 * @param  start：Data address
 * @param  len：data length
 * @param  readbuffer：Stored read data
 * @retval Communication status
 *    @arg 0:Susses
 *    @arg 1:Fail
 ************************************************************************************************************/
uint8_t _BM25S2621_1_RS485_request(uint8_t id, uint8_t funcode, uint16_t start, uint16_t len, uint8_t *readbuffer)
{
  uint16_t rev_num; // Number of received data
  uint8_t i, sendbuffer[9];
  uint8_t crc_check_l, crc_check_h;
  for (i = 0; i < 9; i++)
  {
    sendbuffer[i] = 0;
  }
  sendbuffer[0] = id;
  sendbuffer[1] = funcode;
  sendbuffer[2] = start >> 8;
  sendbuffer[3] = start & 0x00ff;
  sendbuffer[4] = len >> 8;
  sendbuffer[5] = len & 0x00ff;
  _BM25S2621_1_Modbus_CRCcaculate(sendbuffer, 6, &sendbuffer[7], &sendbuffer[6]);

  GPIO_SetOutBits(BM25S2621_1_STA_PIN_GPIO_PORT, BM25S2621_1_STA_PIN_GPIO_PIN);
  BM25S2621_1_writeBytes(sendbuffer, 8);
  GPIO_ClearOutBits(BM25S2621_1_STA_PIN_GPIO_PORT, BM25S2621_1_STA_PIN_GPIO_PIN);

  if (funcode == BM25S2621_1_FUNCODE_READ)
    rev_num = 5 + len * 2;
  else if (funcode == BM25S2621_1_FUNCODE_WRITE)
    rev_num = 8;
  else if (funcode == BM25S2621_1_FUNCODE_TEMPCALI || funcode == BM25S2621_1_FUNCODE_MOISCALI || funcode == BM25S2621_1_FUNCODE_EMPTYWATER)
    rev_num = 7;

  BM25S2621_1_readBytes(readbuffer, rev_num);
  _BM25S2621_1_Modbus_CRCcaculate(readbuffer, rev_num - 2, &crc_check_h, &crc_check_l);
  if (crc_check_h == readbuffer[rev_num - 1] && crc_check_l == readbuffer[rev_num - 2])
  {
    if (funcode == BM25S2621_1_FUNCODE_EMPTYWATER)
    {
      if (readbuffer[4] == 1 || readbuffer[4] == 0xAA)
        return BM25S2621_1_CHECK_OK;
      else
        return BM25S2621_1_CHECK_ERROR;
    }
    else
      return BM25S2621_1_CHECK_OK;
  }
  else
    return BM25S2621_1_CHECK_ERROR;
}
 /***********************************************************************************************************
   * @brief  write data through UART.
   * @param  wbuf:Variables for storing Data to be sent
   * @param  wlen:Length of data sent
   * @retval void
   ************************************************************************************************************/
void BM25S2621_1_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
  UARTM_Write(gBM25S2621_1_SERIAL, wbuf, wlen);
	while(UARTM_IsTxFinished(gBM25S2621_1_SERIAL) == FALSE);
}

/***********************************************************************************************************
 * @brief  read data through UART.
 * @param  rbuf:Variables for storing Data to be obtained
 * @param  rlen:Length of data to be obtained
 * @retval BM25S2621_1_CHECK_OK / BM25S2621_1_TIMEOUT_ERROR
 ************************************************************************************************************/
uint8_t BM25S2621_1_readBytes(uint8_t rbuf[], uint8_t rlen)
{
  uint8_t i = 0, _delayCount = 0;
  for (i = 0; i < rlen; i++)
  {
    _delayCount = 0;
    while (UARTM_GetReadBufferLength(gBM25S2621_1_SERIAL) == 0)
    {
      if (_delayCount > BM25S2621_1_UART_RXTIMEOUT_BYTE)
      {
        return BM25S2621_1_TIMEOUT_ERROR;
      }
      _BM25S2621_1_delayMs(1);
      _delayCount++;
    }
    UARTM_ReadByte(gBM25S2621_1_SERIAL, rbuf + i);
  }
  return 0;
}
/* Private functions ---------------------------------------------------------------------------------------*/
/***********************************************************************************************************
 * @brief  delay ms.
 * @param  count:unit ms
 * @retval void
 ************************************************************************************************************/
void _BM25S2621_1_delayMs(uint32_t count)
{
  count = SystemCoreClock / 8000 * count;
  while (count--)
    ;
}
