/*********************************************************************************************************//**
 * @file    BMD11M134_HT32/src/BMD11M134.c
 * @version V1.0.1
 * @date    2024-08-27
 * @brief   The function of BMD11M134 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMD11M134.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMD11M134_WIRE = BMD11M134_WIRE;
uint32_t gBMD11M134_i2cAddress = 0x2c;
uint8_t BMD11M134_displayArray[20] = {0};
char BMD11M134_displayContent[20] = {0};
uint8_t _BMD11M134_Number = 1;
/*extern I2CMaster_StateTypeDef *pI2CMasterState;*/
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/

/*********************************************************************************************************//**
 * @brief  module wire number select.
 * @param  wire_number: select wire number.
 * @retval BMD11M134_selStatus
                    @arg BMD11M134_SUCCESS
                    @arg BMD11M134_FAILURE 
 ************************************************************************************************************/
BMD11M134_selStatus BMD11M134_selWire(uint32_t wire_number)
{
  if(CHECK_SERIALn(wire_number) == -1)
  {
    return BMD11M134_FAILURE;
  }

  gBMD11M134_WIRE = wire_number;
  return BMD11M134_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using IIC communication.
 * @param  void
 * @retval void       
 ************************************************************************************************************/
void BMD11M134_Init(void)
{
	I2CMaster_Init(gBMD11M134_WIRE, gBMD11M134_i2cAddress, BMD11M134_CLKFREQ);
	_BMD11M134_delay(600);
	_BMD11M134_Number = BMD11M134_getNumber();
}

/*********************************************************************************************************//**
 * @brief  get the number of modules cascaded
 * @param  void
 * @retval BMD11M134_Number: number of modules cascaded
 ************************************************************************************************************/
uint8_t BMD11M134_getNumber(void)
{
  uint8_t BMD11M134_sendBuf[5] = {0x2c,0x01,0x02,0x01,0x00};
	uint8_t BMD11M134_buff[6] = {0};
	uint8_t BMD11M134_Number;
	BMD11M134_sendBuf[4] = BMD11M134_sendBuf[0] + BMD11M134_sendBuf[1]  + BMD11M134_sendBuf[2] + BMD11M134_sendBuf[3];
  _BMD11M134_writeBytes(BMD11M134_sendBuf,5); 
	_BMD11M134_delay(7);
   if(_BMD11M134_readBytes(BMD11M134_buff,6) == BMD11M134_CHECK_OK)
	{
		if(BMD11M134_buff[3] == 0x00)
    {
			BMD11M134_Number = BMD11M134_buff[4];
    }
	}
	_BMD11M134_delay(4);
	 return BMD11M134_Number;
}

/*********************************************************************************************************//**
 * @brief  Clear display
 * @param  void
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_clear(void)
{
	uint8_t BMD11M134_i;
	for (BMD11M134_i = 0; BMD11M134_i < 5*_BMD11M134_Number ; BMD11M134_i++)
	{
		BMD11M134_displayArray[BMD11M134_i] = 0;
	}
	return(BMD11M134_updateDisplay());
}

/*********************************************************************************************************//**
 * @brief  Lights the decimal point of the specified module
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_decimalOnSingle(uint8_t displayNumber)
{
  BMD11M134_displayArray[4 + (displayNumber - 1) * 5] = (BMD11M134_displayArray[4 + (displayNumber - 1) * 5] & 0x01) | 0x02;
  return (BMD11M134_updateDisplay());
}

/*********************************************************************************************************//**
 * @brief  Extinguish the decimal point of the specified module
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_decimalOffSingle(uint8_t displayNumber)
{
  BMD11M134_displayArray[4 + (displayNumber - 1) * 5] = (BMD11M134_displayArray[4 + (displayNumber - 1) * 5] & 0x01) | 0x00;
  return (BMD11M134_updateDisplay());
}

/*********************************************************************************************************//**
 * @brief  Light the decimal point of all the expansion modules
 * @param  void
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_decimalOn(void)
{
 	uint8_t BMD11M134_status;
	uint8_t BMD11M134_i;
	for (BMD11M134_i = 1; BMD11M134_i <= _BMD11M134_Number; BMD11M134_i++)
	{
		BMD11M134_status = BMD11M134_decimalOnSingle(BMD11M134_i);
		if(BMD11M134_status != 0)
		{
			return BMD11M134_status;
		}
	}	
	return BMD11M134_status;	
}

/*********************************************************************************************************//**
 * @brief  Extinguish the decimal points of all expansion modules
 * @param  void
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_decimalOff(void)
{
 	uint8_t BMD11M134_status;
	uint8_t BMD11M134_i;
	for (BMD11M134_i = 1; BMD11M134_i <= _BMD11M134_Number; BMD11M134_i++)
	{
		BMD11M134_status = BMD11M134_decimalOffSingle(BMD11M134_i);
		if(BMD11M134_status != 0)
		{
			return BMD11M134_status;
		}
	}	
	return BMD11M134_status;	
}

/*********************************************************************************************************//**
 * @brief  Lights the colon for the specified module
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_colonOnSingle(uint8_t displayNumber)
{
  BMD11M134_displayArray[4 + (displayNumber - 1) * 5] = (BMD11M134_displayArray[4 + (displayNumber - 1) * 5] & 0x02) | 0x01;
  return (BMD11M134_updateDisplay());
}

/*********************************************************************************************************//**
 * @brief  Extinguish the colon of the specified module
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_colonOffSingle(uint8_t displayNumber)
{
  BMD11M134_displayArray[4 + (displayNumber - 1) * 5] = (BMD11M134_displayArray[4 + (displayNumber - 1) * 5] & 0x02) | 0x00;
  return (BMD11M134_updateDisplay());
}

/*********************************************************************************************************//**
 * @brief  Light colons for all expansion modules
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_colonOn(void)
{
 	uint8_t BMD11M134_status;
	uint8_t BMD11M134_i;
	for (BMD11M134_i = 1; BMD11M134_i <= _BMD11M134_Number; BMD11M134_i++)
	{
		BMD11M134_status = BMD11M134_colonOnSingle(BMD11M134_i);
		if(BMD11M134_status != 0)
		{
			return BMD11M134_status;
		}
	}	
	return BMD11M134_status;	
}

/*********************************************************************************************************//**
 * @brief  Turn off colons of all expansion modules
 * @param  displayNumber: module ID
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_colonOff(void)
{
 	uint8_t BMD11M134_status;
	uint8_t BMD11M134_i;
	for (BMD11M134_i = 1; BMD11M134_i <= _BMD11M134_Number; BMD11M134_i++)
	{
		BMD11M134_status = BMD11M134_colonOffSingle(BMD11M134_i);
		if(BMD11M134_status != 0)
		{
			return BMD11M134_status;
		}
	}	
	return BMD11M134_status;
}

/*********************************************************************************************************//**
 * @brief  The string of the module is moved to the right
 * @param  shiftAmt: Number of moves
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_shiftRight(uint8_t shiftAmt)
{
	uint8_t BMD11M134_Len;
	int BMD11M134_i;
	BMD11M134_Len = _BMD11M134_strlen(BMD11M134_displayContent);
	for (BMD11M134_i = (5 * _BMD11M134_Number) - shiftAmt; BMD11M134_i >= shiftAmt; BMD11M134_i--)
	{
		if(BMD11M134_i != BMD11M134_Len)
		{
			BMD11M134_displayContent[BMD11M134_i] = BMD11M134_displayContent[BMD11M134_i - shiftAmt];
		}
	}
	// Clear the leading characters
	for (BMD11M134_i = 0; BMD11M134_i < shiftAmt; BMD11M134_i++)
	{
		if (BMD11M134_i + shiftAmt > (5 * _BMD11M134_Number))
		{
			break; // Error check
		}
		BMD11M134_displayContent[0 + BMD11M134_i] = ' ';
	}
	return (BMD11M134_printStr(BMD11M134_displayContent));	
}

/*********************************************************************************************************//**
 * @brief  The string of the module is moved to the left
 * @param  shiftAmt: Number of moves
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_shiftLeft(uint8_t shiftAmt)
{
	uint8_t BMD11M134_Len;
	int BMD11M134_i;
	BMD11M134_Len = _BMD11M134_strlen(BMD11M134_displayContent);
	for (BMD11M134_i = 0; BMD11M134_i < BMD11M134_Len; BMD11M134_i++)
	{
		if (BMD11M134_i + shiftAmt > BMD11M134_Len)
		{
			break; // Error check
		}
		if(BMD11M134_i!=BMD11M134_Len)
		{
			BMD11M134_displayContent[BMD11M134_i] = BMD11M134_displayContent[BMD11M134_i + shiftAmt];
		}
	}
	// Clear the trailing characters
	for (BMD11M134_i = 0; BMD11M134_i < shiftAmt; BMD11M134_i++)
	{
		if (BMD11M134_Len - 1 - BMD11M134_i < 0)
		{
			break; //Error check
		}
		BMD11M134_displayContent[BMD11M134_Len - 1 - BMD11M134_i] = '\0';
	}
	return (BMD11M134_printStr(BMD11M134_displayContent));
}

/*********************************************************************************************************//**
 * @brief  Specifies the digit display character for the module
 * @param  displayNumber: module ID
           displayChar: Displays the ASCII code corresponding to the character
           digit: Digital tube number, ranging from 0~3
 * @retval void
 ************************************************************************************************************/
void BMD11M134_printChar(uint8_t displayNumber, uint8_t displayChar, uint8_t digit)
{
	// moved alphanumeric_segs array to PROGMEM
	uint16_t BMD11M134_characterPosition = 65532;

	// space
	if ((displayChar == ' ') || (displayChar == '\0'))
	{
		BMD11M134_characterPosition = 0;
	}
	// Printable Symbols -- Between first character ! and last character ~
	else if (displayChar >= '!' && displayChar <= '~')
	{
		BMD11M134_characterPosition = displayChar - '!' + 1;
	}
	
	if (BMD11M134_characterPosition == 14) // '.'
	{
		BMD11M134_displayArray[4 + (displayNumber - 1) * 5] &=  0x01;
		BMD11M134_displayArray[4 + (displayNumber - 1) * 5] |= 0x02;
	}
	else if (BMD11M134_characterPosition == 26) // ':'
	{
		BMD11M134_displayArray[4 + (displayNumber - 1) * 5] &=  0x02;
		BMD11M134_displayArray[4 + (displayNumber - 1) * 5] |= 0x01;
	}
	else if(BMD11M134_characterPosition == 65532) // unknown character
	{
		BMD11M134_characterPosition = 95;
	}
	
	BMD11M134_displayArray[(displayNumber - 1) * 5 + digit] = BMD11M134_characterPosition;	
}

/*********************************************************************************************************//**
 * @brief  update display
 * @param  void
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_updateDisplay(void)
{
  uint8_t BMD11M134_id;
	uint8_t BMD11M134_i;
	uint8_t BMD11M134_status;
  uint8_t BMD11M134_sendBuf[10] = {0x2c,0x00,0x07,0x02,0x00,0x00,0x00,0x00,0x00,0x00};
	uint8_t BMD11M134_buff[5] = {0};
	 for(BMD11M134_id = 1; BMD11M134_id <= _BMD11M134_Number; BMD11M134_id++)
  {
    BMD11M134_sendBuf[1] = BMD11M134_id;
		for(BMD11M134_i = 0; BMD11M134_i < 5; BMD11M134_i++)
    {
      BMD11M134_sendBuf[BMD11M134_i + 4] = BMD11M134_displayArray[(BMD11M134_id - 1) * 5 + BMD11M134_i];
    }
    BMD11M134_sendBuf[9] = 0;
		for(BMD11M134_i = 0; BMD11M134_i < 9; BMD11M134_i++)
    {
      BMD11M134_sendBuf[9] = BMD11M134_sendBuf[BMD11M134_i] + BMD11M134_sendBuf[9];
    }
		_BMD11M134_delay(4);
		_BMD11M134_writeBytes(BMD11M134_sendBuf,10);
    _BMD11M134_delay(7);
    if(_BMD11M134_readBytes(BMD11M134_buff,5) == BMD11M134_CHECK_OK)
	  {
		 BMD11M134_status = BMD11M134_buff[3];
	  }
  }
	_BMD11M134_delay(10);
	return  BMD11M134_status;
}	

/*********************************************************************************************************//**
 * @brief  display string
 * @param  
           @arg buffer[]:Display character array
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_printStr(char buffer[])
{
  uint8_t BMD11M134_i = 0;
	uint8_t BMD11M134_id = 0;
	uint8_t BMD11M134_digit = 0;

	for(BMD11M134_i = 0; BMD11M134_i < _BMD11M134_strlen(buffer); BMD11M134_i++)
	{
		BMD11M134_displayContent[BMD11M134_i] = buffer[BMD11M134_i];
	}
	BMD11M134_i = 0;
	for(BMD11M134_id = 1; BMD11M134_id <= _BMD11M134_Number; BMD11M134_id++)
	{
		for(BMD11M134_digit = 0; BMD11M134_digit < 4; BMD11M134_digit++)
	    {
			if(BMD11M134_i < _BMD11M134_strlen(buffer))
			{
				if((buffer[BMD11M134_i] == '.') || (buffer[BMD11M134_i] == ':'))
			    {
				    while( (BMD11M134_i < _BMD11M134_strlen(buffer)) && ((buffer[BMD11M134_i] == '.') || (buffer[BMD11M134_i] == ':')))
			        {
						   BMD11M134_printChar(BMD11M134_id , buffer[BMD11M134_i], BMD11M134_digit);
					     BMD11M134_i++;
			        }
				}
				BMD11M134_printChar(BMD11M134_id, buffer[BMD11M134_i], BMD11M134_digit);
				BMD11M134_i++;
			}
			else
			{
				BMD11M134_printChar(BMD11M134_id, '\0', BMD11M134_digit);
			}				
		}			
	}
    return BMD11M134_updateDisplay();	
}

/*********************************************************************************************************//**
 * @brief  Set the brightness of a specified module
 * @param  
           @arg displayNumber: module ID
           @arg brightness: Brightness level
                              0:darker
                              1:normal
                              2:lighter
 * @retval Implementation status
           @arg 0x00: SUCCESS
           @arg 0x40: Check code error
           @arg 0x80: Instruction not supported
           @arg 0xA0: The cascading slave does not answer
 ************************************************************************************************************/
uint8_t BMD11M134_setBrightness(uint8_t displayNumber, uint8_t brightness)
{
  uint8_t BMD11M134_status;
  uint8_t BMD11M134_sendBuf[6] = {0x2c,0x00,0x03,0x03,0x00,0x00};
	uint8_t BMD11M134_buff[5] = {0};
	BMD11M134_sendBuf[1] = displayNumber;
	BMD11M134_sendBuf[4] = brightness;

	BMD11M134_sendBuf[5] = BMD11M134_sendBuf[0] + BMD11M134_sendBuf[1]  + BMD11M134_sendBuf[2] + BMD11M134_sendBuf[3]+ BMD11M134_sendBuf[4];
  _BMD11M134_writeBytes(BMD11M134_sendBuf,6); 
	_BMD11M134_delay(7);
   if(_BMD11M134_readBytes(BMD11M134_buff,5) == BMD11M134_CHECK_OK)
	{
		if(BMD11M134_buff[3] == 0x00)
    {
			BMD11M134_status = BMD11M134_buff[3];
    }
	}
	_BMD11M134_delay(4);
	 return BMD11M134_status;
}

/*********************************************************************************************************//**
 * @brief  Get version number
 * @param  displayNumber: module ID
 * @retval BMD11M134_FWVer: version number
 ************************************************************************************************************/
uint16_t BMD11M134_getFWVer(uint8_t displayNumber)
{
  uint16_t BMD11M134_FWVer;
  uint8_t BMD11M134_sendBuf[5] = {0x2c,0x00,0x02,0x04,0x00};
	uint8_t BMD11M134_buff[7] = {0};
  BMD11M134_sendBuf[1] = displayNumber;
	BMD11M134_sendBuf[4] = BMD11M134_sendBuf[0] + BMD11M134_sendBuf[1]  + BMD11M134_sendBuf[2] + BMD11M134_sendBuf[3];
  _BMD11M134_writeBytes(BMD11M134_sendBuf,5); 
	_BMD11M134_delay(7);
   if(_BMD11M134_readBytes(BMD11M134_buff,7) == BMD11M134_CHECK_OK)
	{
		if(BMD11M134_buff[3] == 0x00)
    {
			BMD11M134_FWVer = (BMD11M134_buff[4]<<8) + BMD11M134_buff[5];
    }
	}
	_BMD11M134_delay(4);
	 return BMD11M134_FWVer;

}

/*********************************************************************************************************//**
 * @brief  Get string length
 * @param  *string: Character string
 * @retval BMD11M134_uLen: string length
 ************************************************************************************************************/
uint8_t _BMD11M134_strlen(char *string)
{
  uint32_t BMD11M134_i;
	uint32_t BMD11M134_uLen = 0;
  for (BMD11M134_i = 0; (string[BMD11M134_i] != '\0')&&(BMD11M134_i < 21); BMD11M134_i++)
  {
     BMD11M134_uLen++;
  }
	return BMD11M134_uLen;
}

/*********************************************************************************************************//**
 * @brief  write data through IIC.
 * @param  wbuf:Variables for storing Data to be sent
 * @param  wlen:Length of data sent
 * @retval void  
 ************************************************************************************************************/
void _BMD11M134_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = gBMD11M134_i2cAddress;
    gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;
    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBMD11M134_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBMD11M134_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
}

/*********************************************************************************************************//**
 * @brief  read data through IIC.
 * @param  rbuf:Variables for storing Data to be obtained
 * @param  rlen:Length of data to be obtained
 * @retval void
 ************************************************************************************************************/
uint8_t _BMD11M134_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	 uint8_t _lenCount = 0;
	 uint8_t _BMD11M134_checkSum = 0;
   I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
   I2CMaster_Typedef I2CMaster = { 0 };

   I2CMaster.uSlaveAddress = gBMD11M134_i2cAddress;
   I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
   I2CMaster.Rx.uLength = rlen;
   I2CMaster.uTimeout_us = 30000;
   I2CMaster.isDynamicOnOff_I2C = FALSE;

   I2CMaster_Read(gBMD11M134_WIRE, &I2CMaster);
	
   do {
     I2CResult = I2CMaster_GetStatus(gBMD11M134_WIRE);
   } while (I2CResult == I2C_MASTER_GOING);
	 
	 //checkSum
  for (_lenCount = 0; _lenCount < (rlen - 1); _lenCount++)
 {
    _BMD11M134_checkSum += rbuf[_lenCount];
  }
  if (_BMD11M134_checkSum == rbuf[rlen - 1])
  {
    return BMD11M134_CHECK_OK; // Check correct
  }
  else
 {
   return BMD11M134_CHECK_ERROR; // Check error
  }
	
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  Delay time,Unit:ms
 * @retval void
 ************************************************************************************************************/
void _BMD11M134_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

/* Private functions ---------------------------------------------------------------------------------------*/
