/*****************************************************************
  File:             BM22S2021-1.cpp
  Author:           BEST MODULES
  Description:      Communication and operation function with module
  History：
  V1.0.1-- initial version；2022-05-26；Arduino IDE : v1.8.13
******************************************************************/
#include  "BM22S2021-1.h"
uint8_t receiveBuffer[41];

/**********************************************************
Description: Select the hardware serial port you need to use
Parameters:       *theSerial：hardware serial 
             Arduino UNO optional:serial(default)
Return:      none       
**********************************************************/
BM22S2021_1::BM22S2021_1(uint8_t statusPin,HardwareSerial*theSerial)
{
  _serial = theSerial;
  _softSerial = NULL;
  _statusPin = statusPin;
}
/**********************************************************
Description: Select the software serial port RX TX you need to use
Parameters:  rxPin:RX pin on the development board
             txPin:TX pin on the development board
Return:      none        
**********************************************************/
BM22S2021_1::BM22S2021_1(uint8_t statusPin,uint8_t rxPin, uint8_t txPin)
{
  _serial = NULL;
  _statusPin = statusPin;
  _rxPin = rxPin;
  _txPin = txPin;
  _softSerial = new SoftwareSerial(_rxPin, _txPin);
}
/**********************************************************
Description: Set serial baud rate
Parameters:  uartBaud：9600(default)
Return:      none
**********************************************************/
void BM22S2021_1::begin()
{
  if (_softSerial != NULL)  {_softSerial->begin(UART_BAUD);}
  else {_serial->begin(UART_BAUD);}
}
/**********************************************************
U0
Description: reset the MCU in the module
Parameters:  none
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::resetModule()
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xAF, 0x00, 0x00, 0x51};
  writeData(sendCmd, 4);
  delay(50);  // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("U0: reset module");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  delay(100); // wait module to executa initialization
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;    
}
/**********************************************************
U1
Description: Query the FW version and production date.
             The FW version number and production date are both 8421 BCD code.
Parameters:  buff(12 bytes): Store the FW version and production date
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::getVerDate(uint8_t buff[])
{
  uint8_t sum=0;  
  uint8_t sendCmd[4] = {0xAD, 0x00, 0x00, 0x53};
  writeData(sendCmd, 4);
  delay(70); //wait module response
  readData(buff);
  if (debug){
    Serial.println("U1: get version and date");
    for (uint8_t i=0; i<12; i++)  {Serial.println(buff[i], HEX);}
  }  
  for (uint8_t i=0; i<11; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[11] == (uint8_t)(~sum+1)) return 0;
  else  return   1;  
}

/**********************************************************
U2
Description:Trigger the air calibration function. The calibration time is 8s. 
Parameters: none
Return: 0: calibration success
        1: failed, command NAK 
        2: calibration failed
NOTE: befor execute calibrateModule(), execute writeRegister(0x2E,0) to disable autoTx
**********************************************************/
uint8_t BM22S2021_1::calibrateModule()
{
  uint8_t sum=0,num=0, errorFlag=0;
  uint8_t sendCmd[4] = {0xAB, 0x00, 0x00, 0x55};
  writeData(sendCmd, 4);
  while(1){
    num=0;sum=0;
    while (num < 8){
      if (_softSerial != NULL) {num = _softSerial->available();}
      else if (_serial != NULL){num = _serial->available();}
    }
    for (uint8_t i=0;i<8;i++){
      if (_softSerial != NULL) {receiveBuffer[i] = _softSerial->read();}
      else if (_serial != NULL){receiveBuffer[i] = _serial->read();}
    }
    for (uint8_t j=0; j<7; j++) {sum += receiveBuffer[j];}   
    if (receiveBuffer[7] != (uint8_t)(~sum+1)){errorFlag++;}
    if (debug){
      Serial.println("U2: calibrate module");
      for (uint8_t i=6; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
    }     
    if ((receiveBuffer[6] == 0xA0)) break;
    else if ((receiveBuffer[6] == 0xF0)) break;
    else if (errorFlag != 0) break;
  }
  if (receiveBuffer[6] == 0xA0) return 0;
  else if (receiveBuffer[6] == 0xF0) return 2;
  else return  1;   
}
/**********************************************************
U3
Description:  Read the current status and data of the module
Parameters:  buff:Store the read data //buff[40-0]/buff[19-0]
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::getInfoPackage(uint8_t buff[])
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xAC, 0x00, 0x00, 0x54};
  writeData(sendCmd, 4);
  delay(100); //wait module response
  readData(buff);
  for (uint8_t i=0; i<39; i++) {sum += buff[i];}  
  if (buff[40] == (uint8_t)(~sum+1)) return 0;
  else return  1;    
}
/**********************************************************
U4
Description:  Write data to the register of the specified address;
Parameters:  
        addr:Destination register address to Write
        Register list:
        1.T0A upper / lower calibration limit (H+L):0X08  0X09
        2.T0B upper / lower calibration limit (H+L):0X0A  0X0B
        3.T0A alarm threshold(H+L):0X10 0X11
        4.T0B alarm threshold(H+L):0X12 0X13
        5. standby smoke detection cycle:0X2D
        6. serial port automatic output:0X2E
        7. alarm output level:0X2F
        data:Data to be written
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::writeRegister(uint8_t addr,uint8_t data)
{
  uint8_t sum = 0;
  uint8_t sendCmd[4]={0xE0,0,0,0};
  sendCmd[1] = addr;
  sendCmd[2] = data;
  sum=sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(60);  //wait module response
  readData(receiveBuffer);
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;    
}
/**********************************************************
U5
Description:  Read the register of the specified address
Parameters:  
        addr:Destination register address to read
        Register list:
        1.T0A upper / lower calibration limit (H+L):0X08  0X09
        2.T0B upper / lower calibration limit (H+L):0X0A  0X0B
        3.T0A alarm threshold(H+L):0X10 0X11
        4.T0B alarm threshold(H+L):0X12 0X13
        5. standby smoke detection cycle:0X2D
        6. serial port automatic output:0X2E
        7. alarm output level:0X2F
        data:Data to be written
Return: data:Store read data
        0xFF:read error
**********************************************************/
uint8_t BM22S2021_1::readRegister(uint8_t addr)
{
  uint8_t sum = 0;
  uint8_t sendCmd[4]={0xD0,0,0,0};
  sendCmd[1] = addr;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);  //wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("U5: read register");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return receiveBuffer[6];
  else return 0xFF;    
}
/**********************************************************
U6
Description: Read the running variable at the specified address
Parameters:  addr:Run variable address
        Run variable list:
        Equipment status :0X90
        VBG voltage a/d value(H+L) :0X91 0X92
        Reference value(H+L) :0X93 0X94
        T0A status (channel a) :0X95
        T0B status (channel B) :0X96
        T0B smoke detection value(H+L) :0X9b  0X9c
        T0B smoke detection value(H+L) :0X9D  0X9E
        T0A calibration zero(H+L) :0X9F 0XA0
        T0B calibration zero(H+L) :0XA1 0XA2
        T0A alarm threshold(H+L) :0XA3  0XA4
        T0B alarm threshold(H+L) :0XA5  0XA6
        A. Variation ratio of channel B :0XA7
        Alarm count :0XAA
        Temperature a/d value(H+L) :0XAB  0XAC
        Fault count :0XAF
        Smoke count :0XB0
        VDD voltage(H+L) :0XB1  0XB2       
Return: data:Store read data 
        0xFF:read error
**********************************************************/
uint8_t BM22S2021_1::readRunningVariables(uint8_t addr)
{
  uint8_t sum = 0;
  uint8_t sendCmd[4]={0xD2,0,0,0};
  sendCmd[1] = addr;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);  //wait module response
  readData(receiveBuffer);
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return receiveBuffer[6];
  else return 0xFF;    
}
/**********************************************************
U7
Description: restore factory default settings
Parameters:  none
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::restoreDefault()
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xA0, 0x00, 0x00, 0x60};
  writeData(sendCmd, 4);
  delay(600); //wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("U7: restore default settings");
    for (uint8_t i=0; i<8; i++)  {Serial.println(receiveBuffer[i], HEX);}
  }  
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;   
}
/**********************************************************
Description: Read the data automatically output by the module
Parameters:  buff: buffer for storing the received data 
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S2021_1::autoRx(uint8_t buff[])
{
  uint8_t sum=0,num=0;
  clear_UART_FIFO();
  while (num < 41){
    if (_softSerial != NULL) {num = _softSerial->available();}
    else if (_serial != NULL){num = _serial->available();}
  }
  for (uint8_t i=0;i<41;i++){
    if (_softSerial != NULL) {buff[i] = _softSerial->read();}
    else if (_serial != NULL){buff[i] = _serial->read();}
  }
  for(uint8_t i=0;i<40;i++){sum += buff[i];}
  sum = ~sum + 1;
  if (debug){
    Serial.print("Auto Rx  ");
    Serial.println(num);
    for(uint8_t i=0;i<40;i++){
      Serial.print(i); Serial.print(": 0x"); Serial.println(buff[i], HEX);
    }
    Serial.println(sum, HEX);
    Serial.println("");  
  }
  if ((sum == buff[40]) && (buff[0]==0xAA)){return 0;}
  else return 1;
}
/**********************************************************
Description: UART readData
Parameters:  buff:Variables for storing Data to be read
             datalength:Length of data plus command
Return:      none
Others:
**********************************************************/
void  BM22S2021_1::readData(uint8_t *buff)
{
  uint8_t i=0;
  if (_softSerial != NULL)
  {
    while(_softSerial->available()>0)
    {
      buff[i] = _softSerial->read();
      i++;
    }
  }
  else
  {
    while(_serial->available()>0)
    {
      buff[i] = _serial->read();
      i++;
    }
  }
}
/**********************************************************
Description: UART writeData
             Automatically determine whether a hardware or software serial  is used
Parameters:  buff:Variables for storing Data to be read
             command:Command to read data
             datalength:Length of data plus command
Return:      none
Others:
**********************************************************/
void  BM22S2021_1::writeData(uint8_t buff[], uint8_t datalength)
{
  clear_UART_FIFO();
  if (_softSerial != NULL)
  {
    _softSerial->write(buff, datalength);
  }
  else
  {
    _serial->write(buff, datalength);
  }
}
/**********************************************************
Description: Clear UART Receive FIFO
Parameters: None
Return: None
Others: None
**********************************************************/
void BM22S2021_1::clear_UART_FIFO()
{
  if (_softSerial != NULL)
  {
    while (_softSerial->available() > 0) {_softSerial->read();}
  }
  if (_serial != NULL)
  {
    while (_serial->available() > 0) {_serial->read();}
  }
}
