/************************************************************
* Infrared Temperature Sensing Module BMH06203 example code
* Arduino IDE version: 1.8.13
* Arduino Board: UNO R3
* Wire Connection:
*   BMH06203 module pin 1 (VDD) connects to UNO pin 5V
*   BMH06203 module pin 2 (SCL) connects to UNO pin SCL (or A5)
*   BMH06203 module pin 3 (SDA) connects to UNO pin SDA (or A4)
*   BMH06203 module pin 4 (GND) connects to UNO pin GND

* Created on 2021/9/29 by Best Modules Corp.
*************************************************************/

#include <Wire.h> 

// I2C address
#define SLAVE_ADDRESS 0x28        // I2C address of this module

// RAM command
#define AMB_TEMP 0x08              // read Ambient temperature 
#define OBJ_TEMP 0x09              // read Object temperature 
#define BODY_TEMP 0x0A             // read Human Body temperature

// EEPROM command
#define MODE_CMD 0x28             // Mode choose
#define EMISSIVITY_CMD 0x29       // Emissivity setting
#define PWM_MIN_CMD 0x2A          // Set the minimum temperature in PWM mode(default 0, unit 0.1°C)
#define PWM_MAX_CMD 0x2B          // Set the maximum temperature in PWM mode(default 1000, unit 0.1°C)
#define IO_CMD 0x2C               // Set the temperature threshold in IO mode(default 0, unit 0.1°C)
#define SLEEP_CMD 0xFF            // Sleep mode setting
#define SLEEP_DATA 0x1234         // Sleep mode data

// define return value when a READ command error (PEC code error)
#define READ_ERROR 0xFFFF

void setup() 
{
  Wire.begin();
  Serial.begin(9600);  
  /*************************************************
  Description: Set the emissivity to 95 (= 0x5F)
  Note: need to power off then power on to let the command take effect
  *************************************************/ 
  I2CWrite(EMISSIVITY_CMD,0x5F);

  /*************************************************
  Description: Set to pwm mode(maximum=1000,minimum=100)
  Note: need to power off then power on to let the command take effect
  *************************************************/
  // I2CWrite(PWM_MIN_CMD,0x0064);
  // I2CWrite(PWM_MAX_CMD,0x03E8);
  // I2CWrite(MODE_CMD,0x01);
  
  /*************************************************
  Description: Set to I/O mode(threshold=34.0°C)
  Note: need to power off then power on to let the command take effect
  *************************************************/  
  // I2CWrite(IO_CMD,0x0154);
  // I2CWrite(MODE_CMD,0x02);
  
  /*************************************************
  Description: Change to I2C mode when in PWM/IO mode
   *************************************************/  
  // digitalWrite(A5,0); // set SCL to low for 50ms to become I2C mode
  // delay(50);
  // I2CWrite(MODE_CMD,0x00); // set I2C mode
}

void loop() 
{
  Serial.print("Emissivity : ");
  Serial.println(I2CRead(EMISSIVITY_CMD));
  delay(1000);
  
  Serial.print("Ambient temperature : ");
  Serial.println(I2CRead(AMB_TEMP));
  delay(1000);
  
  Serial.print("Object temperature : ");
  Serial.println(I2CRead(OBJ_TEMP));
  delay(1000);
  
  Serial.print("Body temperature : ");
  Serial.println(I2CRead(BODY_TEMP));
  delay(1000);
}

/*************************************************
Description: Read command from RAM/EEPROM
Parameter:          
  command: the command to be executed

Return:             
  command is AMB_TEMP : return ambient temperature value(unit 0.1°C)
  command is OBJ_TEMP : return object temperature value(unit 0.1°C)
  command is BODY_TEMP : return body temperature value(unit 0.1°C)
  command is MODE_CMD : return modeconfig value
  command is EMISSIVITY_CMD : return emissivity value
  command is PWM_MIN_CMD : return minimum temperature value(default 0, unit 0.1°C)
  command is PWM_MAX_CMD : return mmaximum temperature value(default 1000, unit 0.1°C)
  command is IO_CMD : return I/O threshold value(default 0, unit 0.1°C)         
*************************************************/
uint16_t I2CRead(uint8_t command)
{    
      uint8_t Buffer[3],counter=0;
      uint16_t pec,data;
      Wire.beginTransmission(SLAVE_ADDRESS);   // Send start command and Slave Address
      Wire.write(command); 
      Wire.endTransmission(false);     // transmission with an I2C restart   
      delayMicroseconds(400);          // time required before reading data 
      Wire.requestFrom(SLAVE_ADDRESS,3);       // Send a request to return 3 bytes data from a slave device
      while(Wire.available())
      {
        Buffer[counter++] = Wire.read();  // Returns the number of bytes available for retrieval with read()
        if(counter==3)break;
      }
      data=(Buffer[1]<<8)+Buffer[0];           // Data =  (Data_HighByte << 8) + Data_LowByte;  
      pec=Buffer[2];                           // pec_r = (Buffer[1]+Buffer[0]) & 0xFF
      if(pec == (Buffer[1]+Buffer[0])&0xFF)return data;  
      else return READ_ERROR; 
}

/*************************************************
Description: Write command to EEPROM
Parameter:          
  command: The command to be executed
  value: The value to be written
Note: 
  Set modeconfig : command => MODE_CMD (default is I2C)
                   value   => 0x00 (Set the module to I2C mode)     
                   value   => 0x01 (Set the module to PWM mode)     
                   value   => 0x02 (Set the module to I/O mode (CTR=0))   
                              CTR = 0 : IO output low level when ≥temperature threshold, IO output high level when <temperature threshold  
                   value   => 0x06 (Set the module to I/O mode (CTR=1))   
                              CTR = 1 : IO output High level when ≥temperature threshold, IO output low level when <temperature threshold  
                      
  Set emissivity : command => EMISSIVITY_CMD
                   value   => Emissivity value
                   * If the emissivity is ≥100, the emissivity is 100%
                   * If the emissivity is 95, the emissivity is 95%
                   * If the emissivity is 50, the emissivity is 50%
                   
  Set sleep mode : command => SLEEP_CMD
                   value   => SLEEP_DATA
                   
--------------------------If modeconfig choose PWM--------------------------
  Set PWM minimum : command => PWM_MIN_CMD
                    value   => Minimum temperature value
  Set PWM maximum : command => PWM_MAX_CMD
                    value   => Mmaximum temperature value
                    * If set PWM minimum to 100, PWM maximum to 1000,the temperature is 35°C at this time, 
                    * the PWM output is (350-100) ➗ (1000-100) = 28% duty cycle Square wave.
                   
--------------------------If modeconfig choose IO--------------------------- 
  Set I/O mode : command => IO_CMD
                   value => I/O threshold value
                   * If the temperature threshold IO is set to 450,the level will change when the temperature is ≥45.0°C                 
*************************************************/
void I2CWrite(uint8_t command ,uint16_t value)
{
      uint8_t valueH=(uint8_t)(value>>8);       // High Byte of value
      uint8_t valueL=(uint8_t)(value&0xFF);     // Low Byte of value
      Wire.beginTransmission(SLAVE_ADDRESS);    // Send start cmd and Slave Address
      Wire.write(command);   
      Wire.write(valueL);   
      Wire.write(valueH);   
      Wire.write((command + valueL + valueH)&0xFF);  // pec_w = (Command + Data_LowByte + HighByte) & 0xFF;
      Wire.endTransmission();          // transmission with an I2C stop  
      delay(6);            // time required for writing data to EEPROM
}
