/*********************************************************************************************************//**
 * @file    BMH06203_HT32/src/BMH06203.c
 * @version V1.0.1
 * @date    2024-08-27
 * @brief   The function of BMH06203 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/



/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMH06203.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMH06203_WIRE =BMH06203_WIRE;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief 	module wire number select.
 * @param 	wire_number: select wire number.
 * @retval 	BMH06203_SUCCESS/BMH06203_FAILURE
 ************************************************************************************************************/
BMH06203_selStatus BMH06203_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BMH06203_FAILURE;
  }

  gBMH06203_WIRE = wire_number;
  return BMH06203_SUCCESS;
}
/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BMH06203_Init(void)
{
	I2CMaster_Init(gBMH06203_WIRE, BMH06203_MASTERADDR, BMH06203_CLKFREQ);
}
/*********************************************************************************************************//**
 * @brief  	Get temperature data
 * @param 	TYPE
						@arg 0x08(BMH06203_AMB_TEMP):Environment temperature(unit )
						@arg 0x09(BMH06203_OBJ_TEMP):Surface temperature(unit )
						@arg 0x0A(BMH06203_BODY_TEMP):Body temperature(unit )
 * @retval 	Temperature datauint:
 ************************************************************************************************************/
float BMH06203_readTemperature(uint8_t TYPE)
{
	uint8_t Tempbuf[3];
	uint8_t type[1]={0};
	type[0] = TYPE;
	
	BMH06203_writeBytes(type,1);
	if(BMH06203_readBytes(Tempbuf,3) ==BMH06203_READ_OK)
	{
		_BMH06203_delay(10);
		return ((float)(Tempbuf[1] * 256 + Tempbuf[0])) / 10;
	}
	_BMH06203_delay(10);
	return 0;  
}
/*********************************************************************************************************//**
 * @brief  	Enter halt mode
 * @param 	void
 * @retval 	void
 ************************************************************************************************************/
void BMH06203_sleep(void)
{
	uint8_t haltbuf[4] = {0xFF,0x34,0x12,0x45};
	BMH06203_writeBytes(haltbuf,4);
}
/*********************************************************************************************************//**
 * @brief  	Write EEPROM
 * @param 	addr:EEPROM address 
						data:Data written into EEPROM
 * @retval 	void
 ************************************************************************************************************/
void BMH06203_writeEEPROM(uint8_t addr, uint16_t data)
{
	uint8_t EEPROMPWD[4] = {0};
	
	EEPROMPWD[0] = 0x20 + addr;
	EEPROMPWD[1] = data & 0x00ff;
	EEPROMPWD[2] = data >> 8;
	EEPROMPWD[3] = (EEPROMPWD[0] + EEPROMPWD[1] + EEPROMPWD[2]) & 0xff;
	BMH06203_writeBytes(EEPROMPWD,4);
}
/*********************************************************************************************************//**
 * @brief  	Read EEPROM value
 * @param 	addr:EEPROM address 
 * @retval 	EEPROM address value
 ************************************************************************************************************/
uint16_t BMH06203_readEEPROM(uint8_t addr)
{
	uint8_t readcommand[1] = {0};
	uint8_t readbuf[3] = {0};	
	
	readcommand[0] = 0x20+addr;
	BMH06203_writeBytes(readcommand,1);

	if(BMH06203_readBytes(readbuf,3) == BMH06203_READ_OK)
	{
		_BMH06203_delay(10);
		return readbuf[1] * 256 + readbuf[0];
	}
	else
	{
		_BMH06203_delay(10);
		return 0;
	}
}
/*********************************************************************************************************//**
 * @brief  	Set the output mode
 * @param 	mode:Module working mode
						@arg 0x00(BMH06203_IIC_MODE):IIC MODE
						@arg 0x01(BMH06203_PWM_MODE):PWM MODE
						@arg 0x02(BMH06203_IO_MODE1):IO MODE 1
						@arg 0x06(BMH06203_IO_MODE2):IO MODE 2
 * @retval 	void
 ************************************************************************************************************/
void BMH06203_setMode(uint8_t Mode)
{
	_BMH06203_SCL_Init();
	_BMH06203_SCL_LOW();
	_BMH06203_delay(100);//Wait 50ms and then power cycle up58
	_BMH06203_SCL_HIGH();
	BMH06203_Init();
	BMH06203_writeEEPROM(0x08,Mode);
}
/*********************************************************************************************************//**
 * @brief  	Maximum and minimum temperature values when setting PWM mode
 * @param 	min:Set the minimum temperature 
						max:Set the maximum temperatur
 * @retval 	void
 ************************************************************************************************************/
void BMH06203_setPWMParam(uint16_t min,uint16_t max)
{
	uint16_t _min = min *10;
  uint16_t _max = max *10;
  BMH06203_writeEEPROM(0x0A,_min); 
  BMH06203_writeEEPROM(0x0B,_max); 
}
/*********************************************************************************************************//**
 * @brief  	Temperature threshold when setting IO mode
 * @param 	threshold:Threshold
 * @retval 	void
 ************************************************************************************************************/
void BMH06203_setIOParam(uint16_t threshold)
{
	uint16_t _threshold = threshold * 10;
  BMH06203_writeEEPROM(0x0C,_threshold);
}
/*********************************************************************************************************//**
 * @brief  	write data through IIC
 * @param  	wbuf:Variables for storing Data to be sent
						wlen:Length of data sent
 * @retval 	void  
 ************************************************************************************************************/
void BMH06203_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
	I2CMaster_Typedef gI2CMaster = {0x00};
	
	gI2CMaster.uSlaveAddress = BMH06203_MASTERADDR;
	gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
	gI2CMaster.Tx.uLength = wlen;
	gI2CMaster.uTimeout_us = 30000;
	gI2CMaster.isDynamicOnOff_I2C = FALSE;

	I2CResult = I2C_MASTER_GOING;
	I2CMaster_Write(gBMH06203_WIRE,&gI2CMaster);
	do {
		I2CResult = I2CMaster_GetStatus(gBMH06203_WIRE);
	} while (I2CResult == I2C_MASTER_GOING);
	_BMH06203_delay(10);
}
/*********************************************************************************************************//**
 * @brief  	read data through IIC
 * @param  	rbuf:Variables for storing Data to be obtained
						rlen:Length of data to be obtained
 * @retval 	BMH06203_READ_OK/BMH06203_READ_ERROR/BMH06203_NO_ACK/BMH06203_TIMEOUT_ERROR 
 ************************************************************************************************************/
uint8_t BMH06203_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	uint8_t i = 0;
	uint8_t checkSum = 0;

	I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
	I2CMaster_Typedef I2CMaster = { 0 };

	I2CMaster.uSlaveAddress = BMH06203_MASTERADDR;
	I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
	I2CMaster.Rx.uLength = rlen;
	I2CMaster.uTimeout_us = 30000;
	I2CMaster.isDynamicOnOff_I2C = FALSE;

	I2CMaster_Read(gBMH06203_WIRE, &I2CMaster);
	
	do {
		I2CResult = I2CMaster_GetStatus(gBMH06203_WIRE);
	} while (I2CResult == I2C_MASTER_GOING);

	_BMH06203_delay(1);
	for(i = 0;i < rlen - 1;i++)
	{
		checkSum = checkSum + rbuf[i];
	}
	if(rbuf[i] != checkSum)
	{
		return BMH06203_READ_ERROR;
	}
	return BMH06203_READ_OK;
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BMH06203_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}
/*********************************************************************************************************//**
 * @brief  SCL pin initialization.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void _BMH06203_SCL_Init(void)
{
	#if defined(USE_BM53A367A_DVB)
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{ 0 }};
	CKCUClock.Bit.PC = 1;
	CKCU_PeripClockConfig(CKCUClock, ENABLE);
	
	switch(BMH06203_WIRE)
	{
		case 0:
		{
			AFIO_GPxConfig(GPIO_PC, AFIO_PIN_4, AFIO_FUN_GPIO);
			GPIO_PullResistorConfig(HT_GPIOC, GPIO_PIN_4, GPIO_PR_UP);
			GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_4, SET);
			GPIO_DirectionConfig(HT_GPIOC, GPIO_PIN_4, GPIO_DIR_OUT);
			break;
		}
		case 1:
		{
			AFIO_GPxConfig(GPIO_PC, AFIO_PIN_12, AFIO_FUN_GPIO);
			GPIO_PullResistorConfig(HT_GPIOC, GPIO_PIN_12, GPIO_PR_UP);
			GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_12, SET);
			GPIO_DirectionConfig(HT_GPIOC, GPIO_PIN_12, GPIO_DIR_OUT);
			break;
		}
		case 2:
		{
			AFIO_GPxConfig(GPIO_PC, AFIO_PIN_14, AFIO_FUN_GPIO);
			GPIO_PullResistorConfig(HT_GPIOC, GPIO_PIN_14, GPIO_PR_UP);
			GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_14, SET);
			GPIO_DirectionConfig(HT_GPIOC, GPIO_PIN_14, GPIO_DIR_OUT);
			break;
		}
		default:break;
	}
	#endif
	
	#if defined(USE_HT32F52352_SK)
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{ 0 }};
	CKCUClock.Bit.PA = 1;
	CKCU_PeripClockConfig(CKCUClock, ENABLE);
	

	AFIO_GPxConfig(GPIO_PA, AFIO_PIN_0, AFIO_FUN_GPIO);
	GPIO_PullResistorConfig(HT_GPIOA, GPIO_PIN_0, GPIO_PR_UP);
	GPIO_WriteOutBits(HT_GPIOA, GPIO_PIN_0, SET);
	GPIO_DirectionConfig(HT_GPIOA, GPIO_PIN_0, GPIO_DIR_OUT);
	#endif
	
}
/*********************************************************************************************************//**
 * @brief  SCL pin outputs high level.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void _BMH06203_SCL_HIGH(void)
{
	#if defined(USE_BM53A367A_DVB)
	switch(BMH06203_WIRE)
	{
		case 0:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_4, SET);break;
		case 1:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_12, SET);break;
		case 2:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_14, SET);break;
		default:break;
	}
	#endif
	 
	#if defined(USE_HT32F52352_SK)
	GPIO_WriteOutBits(HT_GPIOA, GPIO_PIN_0, SET); 
	#endif
}
/*********************************************************************************************************//**
 * @brief  SCL pin output low level.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void _BMH06203_SCL_LOW(void)
{
	#if defined(USE_BM53A367A_DVB)
	switch(BMH06203_WIRE)
	{
		case 0:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_4, 	RESET);break;
		case 1:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_12, RESET);break;
		case 2:GPIO_WriteOutBits(HT_GPIOC, GPIO_PIN_14, RESET);break;
		default:break;
	}
	#endif
	
	#if defined(USE_HT32F52352_SK)
	GPIO_WriteOutBits(HT_GPIOA, GPIO_PIN_0, RESET);
	#endif
}
/* Private functions ---------------------------------------------------------------------------------------*/
