/*********************************************************************************************************//**
 * @file    BME82M131_HT32/src/BME82M131.c
 * @version V1.0.1
 * @date    2024-09-06
 * @brief   The function of BME82M131 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "BME82M131.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBME82M131_WIRE = BME82M131_WIRE;
/* Private variables ---------------------------------------------------------------------------------------*/
uint8_t _BME82M131_tx_buf[10] = {0};
uint8_t _BME82M131_rx_buf[10] = {0};
uint8_t _BME82M131_SumOfModules = 1;
typedef struct
{
	union
	{
		uint16_t all;
		struct
		{	
			uint16_t ALS_SD : 1; 	 // ALS shut down setting
			uint16_t ALS_INT_EN : 1; // ALS interrupt enable setting
			uint16_t ALS_RES1 : 2; 	 // Reserved
			uint16_t ALS_PERS : 2; 	 // ALS persistence protect number setting
			uint16_t ALS_IT : 4; 	 // ALS integration time setting
			uint16_t ALS_RES2 : 1; 	 // Reserved
			uint16_t ALS_GAIN : 2; 	 // ALS GAIN selection
			uint16_t ALS_RES3 : 3; 	 // Reserved
		};
	};
} BME82M131_CONF_REG_t;
BME82M131_CONF_REG_t _BME82M131_ConfReg;
		
/** Provide bit field access to the configuration register */
typedef struct
{
	union
	{
		uint16_t all;
		struct
		{	
			uint16_t PSM_EN: 1; 	 // ALS shut down setting
			uint16_t PSM_MODE : 2; // ALS interrupt enable setting
			uint16_t PSM_RES : 13;
		};
	};
} BME82M131_PSM_REG_t;
BME82M131_PSM_REG_t _BME82M131_PsmReg;
		
/** Provide bit field access to the interrupt status register
	Note: reading the interrupt status register clears the interrupts.
				So, we need to check both interrupt flags in a single read. */
typedef struct
{
	union
	{
		uint16_t all;
		struct
		{
			uint16_t INT_STATUS_RES : 14; // Reserved
			// Bit 14 indicates if the high threshold was exceeded
			// Bit 15 indicates if the low threshold was exceeded
			uint16_t INT_STATUS_FLAGS : 2;
		};
	};
} BME82M131_INT_STATUS_REG_t;	

/** The sensor resolution vs. gain and integration time. Taken from the BME82M131 Application Note. */
const float BME82M131_LUX_RESOLUTION[4][6] =
{
	// 25ms    50ms    100ms   200ms   400ms   800ms
	{0.2304, 0.1152, 0.0576, 0.0288, 0.0144, 0.0072}, // Gain 1
	{0.1152, 0.0576, 0.0288, 0.0144, 0.0072, 0.0036}, // Gain 2
	{1.8432, 0.9216, 0.4608, 0.2304, 0.1152, 0.0576}, // Gain 1/8
	{0.9216, 0.4608, 0.2304, 0.1152, 0.0576, 0.0288}  // Gain 1/4
};
/** The BME82M131 gain (sensitivity) settings as text (string) */
const char *BME82M131_GAIN_SETTINGS[5] =
{
 // Note: these are in the order defined by ALS_SM and sensitivity_mode_t
 "x1","x2","x1/8","x1/4","INVALID"
};
/** The BME82M131 integration time settings as text (strting) */
const char *BME82M131_INTEGRATION_TIMES[7] =
{
"25ms","50ms","100ms","200ms","400ms","800ms","INVALID"
};
/** The BME82M131 persistence protect settings as text (string) */
		const char *BME82M131_PERSISTENCE_PROTECT_SETTINGS[5] =
		{
	"1", "2", "4", "8", "INVALID"
		};
/** The BME82M131 power saving mode seting as text (string) */
		const char *BME82M131_POWER_SAVING_MODE_SETTINGS[5] =
		{
	"MODE1", "MODE2", "MODE3", "MODE4", "INVALID"
		};	

const uint8_t _BME82M131_CMD_CHECK_MODULE = 0x01;
const uint8_t _BME82M131_CMD_W_ALS_CONFIG = 0x02;
const uint8_t _BME82M131_CMD_W_HIGH_THRESHOLD = 0x03;
const uint8_t _BME82M131_CMD_W_LOW_THRESHOLD = 0x04;
const uint8_t _BME82M131_CMD_W_POWER_SAVING_MODE = 0x05;
const uint8_t _BME82M131_CMD_R_ALS_CONFIG = 0x06;
const uint8_t _BME82M131_CMD_R_HIGH_THRESHOLD = 0x07;
const uint8_t _BME82M131_CMD_R_LOW_THRESHOLD = 0x08;
const uint8_t _BME82M131_CMD_R_POWER_SAVING_MODE = 0x09;
const uint8_t _BME82M131_CMD_R_ALS_OUTPUT = 0x0A;
const uint8_t _BME82M131_CMD_R_WHITE_OUTPUT = 0x0B;
const uint8_t _BME82M131_CMD_R_INT_STATUS = 0x0C;
/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief module wire number select.
 * @param wire_number: select wire number.
 * @retval BME82M131_SUCCESS/BME82M131_FAILURE
 ************************************************************************************************************/
BME82M131_selStatus BME82M131_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BME82M131_FAILURE;
  }

  gBME82M131_WIRE = wire_number;
  return BME82M131_SUCCESS;
}

/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BME82M131_Init(void)
{
	_BME82M131_delay(500);
	I2CMaster_Init(gBME82M131_WIRE, BME82M131_MASTERADDR, BME82M131_CLKFREQ);

	//INPUT:INT Pin
	{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
  CKCUClock.Bit.BME82M131_INTPIN_GPIO = 1;
  CKCUClock.Bit.AFIO       = 1;
  CKCU_PeripClockConfig(CKCUClock, ENABLE);
	}
	AFIO_GPxConfig(BME82M131_INTPIN_GPIO_ID, BME82M131_INTPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_DirectionConfig(BME82M131_INTPIN_GPIO_PORT, BME82M131_INTPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_PullResistorConfig(BME82M131_INTPIN_GPIO_PORT, BME82M131_INTPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_InputConfig(BME82M131_INTPIN_GPIO_PORT, BME82M131_INTPIN_GPIO_PIN, ENABLE);
	
	//others
	BME82M131_getNumber();
	BME82M131_initialize();
}

/*********************************************************************************************************//**
 * @brief  get INT PIN Status
 * @param  void
 * @retval INT PIN Status
             @arg 1:INT status is HIGH(default)
             @arg 0:INT status is LOW(get a interrupt)
 ************************************************************************************************************/
uint8_t BME82M131_getINT()
{
	return (GPIO_ReadInBit(BME82M131_INTPIN_GPIO_PORT, BME82M131_INTPIN_GPIO_PIN));
}

/*********************************************************************************************************//**
 * @brief  Gets the number of modules
 * @param  void
 * @retval _BME82M131_rx_buf[4]: Number of connecting modules 
 ************************************************************************************************************/
uint8_t BME82M131_getNumber()
{
	uint8_t BME82M131_status;
	_BME82M131_tx_buf[3] = _BME82M131_CMD_CHECK_MODULE;	
	_BME82M131_delay(20);
  BME82M131_sendData(1, 5, _BME82M131_tx_buf);
	_BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,6))
	{
  BME82M131_status=_BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      _BME82M131_SumOfModules = _BME82M131_rx_buf[4];
      return _BME82M131_rx_buf[4];
    }
    else
    {
      return 0;
    }
	}
  else
  {
	 return 0;
  }
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's ambient light sensor data (ALS)
 * @param  sensornumber : Module Number
 * @retval als:ambient light data
 ************************************************************************************************************/
uint16_t BME82M131_readALS(uint8_t sensornumber)
{
  uint8_t BME82M131_status; 
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_ALS_OUTPUT;
  _BME82M131_delay(20);
  BME82M131_sendData(sensornumber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(20);
   if(BME82M131_readBytes(_BME82M131_rx_buf,7))
   {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status == 0)
    {
      uint16_t als=0;
      als = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      return als;
    }
    else
    {
      return 0;
    }    
   }
   else
   {
    return 0;
   }  
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's white level data (WHITE)
 * @param  sensornumber : Module Number
 * @retval als:white level data
 ************************************************************************************************************/
uint16_t BME82M131_readWhite(uint8_t sensornumber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_white=0;
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_WHITE_OUTPUT;

  _BME82M131_delay(60);
  BME82M131_sendData(sensornumber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(60);
  if(BME82M131_readBytes(_BME82M131_rx_buf,7))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status == 0)
    {
      BME82M131_white = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
     return BME82M131_white;
    }
    else
    {
      return 0;
    } 
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Read the sensor data and calculate the lux
 * @param  sensorNmuber : Module Number
 * @retval lux:Floating point Lux data
 ************************************************************************************************************/
float BME82M131_readLux(uint8_t sensorNmuber)
{
  BME82M131_GAIN_t BME82M131_gain = BME82M131_GAIN_2;
  BME82M131_IT_TIME_t BME82M131_it = BME82M131_IT_800ms;
  uint16_t BME82M131_als = 0;
  float BME82M131_lux = 0;
  float BME82M131_resolution; 
	uint8_t BME82M131_gain_flag=0;
	uint8_t BME82M131_integration_flag = 0;
  BME82M131_gain_flag=BME82M131_getGain(sensorNmuber);
  if(BME82M131_gain_flag==0)
  {
    BME82M131_gain=BME82M131_GAIN_1;
  }
  else if(BME82M131_gain_flag==1)
  {
     BME82M131_gain=BME82M131_GAIN_2;
  }
 else if(BME82M131_gain_flag==2)
  {
     BME82M131_gain=BME82M131_GAIN_1_8;
  }
 else if(BME82M131_gain_flag==3)
  {
     BME82M131_gain=BME82M131_GAIN_1_4;
  }
  else if(BME82M131_gain_flag==4)
  {
    return 0;
  }
  BME82M131_integration_flag = BME82M131_getIntegrationTime(sensorNmuber);  
  if(BME82M131_integration_flag==0)
  {
   BME82M131_it= BME82M131_IT_25ms;
  }
  else if(BME82M131_integration_flag==1)
  {
    BME82M131_it= BME82M131_IT_50ms;
  }
  else if(BME82M131_integration_flag==2)
  {
    BME82M131_it= BME82M131_IT_100ms;
  }
  else if(BME82M131_integration_flag==3)
  {
    BME82M131_it= BME82M131_IT_200ms;
  }
  else if(BME82M131_integration_flag==4)
  {
    BME82M131_it= BME82M131_IT_400ms;
  }
  else if(BME82M131_integration_flag==5)
  {
    BME82M131_it= BME82M131_IT_800ms;
  }
  else if(BME82M131_integration_flag==6)
  {
    return 0;
  }
  BME82M131_als = BME82M131_readALS(sensorNmuber);
  BME82M131_resolution = BME82M131_LUX_RESOLUTION[BME82M131_gain][BME82M131_it];
  BME82M131_lux = (float)BME82M131_als * BME82M131_resolution;
  return BME82M131_lux;
}

/*********************************************************************************************************//**
 * @brief  get the VEML7700's interrupt status register
 * @param  sensorNmuber : Module Number
 * @retval InterruptFalg  
            @arg  0 : BME82M131_INT_STATUS_NONE,
            @arg  1 : BME82M131_INT_STATUS_HIGH,
            @arg  2 : BME82M131_INT_STATUS_LOW,
            @arg  3 : BME82M131_INT_STATUS_BOTH
 ************************************************************************************************************/
uint8_t BME82M131_getInterruptFalg(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  BME82M131_INT_STATUS_t  BME82M131_it_status;
  BME82M131_INT_STATUS_REG_t BME82M131_REG_Temp;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_INT_STATUS;

  _BME82M131_delay(60);
  BME82M131_sendData(sensorNmuber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(60);
  if(BME82M131_readBytes( _BME82M131_rx_buf,7))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status == 0)
    {
      BME82M131_REG_Temp.all = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      BME82M131_it_status = (BME82M131_INT_STATUS_t)BME82M131_REG_Temp.INT_STATUS_FLAGS;
      if(BME82M131_it_status==BME82M131_INT_STATUS_NONE)  
      {
        return 0; 
      }
      else if(BME82M131_it_status==BME82M131_INT_STATUS_HIGH)  
      {
        return 1; 
      }
      else if(BME82M131_it_status==BME82M131_INT_STATUS_LOW)  
      {
        return 2; 
      }
      else if(BME82M131_it_status==BME82M131_INT_STATUS_BOTH)  
      {
        return 3; 
      }
    }
    else
    {
     return 0;
    }
  }
  else
  {
     return 0;
  } 
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's Power saving mode.
 * @param  sensorNmuber : Module Number
 * @retval PowerSavingMode  
            @arg  0 : BME82M131_POWER_SAVING_MODE1  
            @arg  1 : BME82M131_POWER_SAVING_MODE2    
            @arg  2 : BME82M131_POWER_SAVING_MODE3    
            @arg  3 : BME82M131_POWER_SAVING_MODE4
 ************************************************************************************************************/
uint8_t BME82M131_getPowerSavingMode(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_psm;
  BME82M131_PSM_MODE_t  BME82M131_mode;
  BME82M131_status = BME82M131_getPsmReg(sensorNmuber, &BME82M131_psm);
  if(BME82M131_status)
  {
    _BME82M131_PsmReg.all = BME82M131_psm;
    BME82M131_mode = (BME82M131_PSM_MODE_t)_BME82M131_PsmReg.PSM_MODE;
    if(BME82M131_mode == BME82M131_POWER_SAVING_MODE1)
    {
      return 0;
    }
    else if(BME82M131_mode == BME82M131_POWER_SAVING_MODE2)
    {
      return 1;
    }
    else if(BME82M131_mode == BME82M131_POWER_SAVING_MODE3)
    {
      return 2;
    }
    else if(BME82M131_mode == BME82M131_POWER_SAVING_MODE4)
    {
      return 3;
    }
  }
  else
  {
    return 0;
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's power saving mode enable state.
 * @param  sensorNmuber : Module Number
 * @retval PowerSavingModeStatus 
            @arg  0-BME82M131_POWER_SAVING_MODE_DISABLE      
            @arg  1-BME82M131_POWER_SAVING_MODE_ENABLE   
 ************************************************************************************************************/
uint8_t BME82M131_getPowerSavingModeStatus(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_psm;
  BME82M131_PSMEN_t BME82M131_psmen;
  BME82M131_status = BME82M131_getPsmReg(sensorNmuber, &BME82M131_psm);
  if(BME82M131_status)
  {
    _BME82M131_PsmReg.all = BME82M131_psm;
    BME82M131_psmen = (BME82M131_PSMEN_t)_BME82M131_PsmReg.PSM_EN;
    if(BME82M131_psmen==BME82M131_POWER_SAVING_MODE_DISABLE)
    {
      return 0;
    }
    else if(BME82M131_psmen==BME82M131_POWER_SAVING_MODE_ENABLE)
    {
      return 1;
    }
  }
  else
  {
    return 0;
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's moudle status (ALS_SD)
 * @param  sensorNmuber : Module Number
 * @retval MoudleStatus
            @arg  0-BME82M131_POWER_ON      
            @arg  1-BME82M131_SHUT_DOWN  
 ************************************************************************************************************/
uint8_t BME82M131_getMoudleStatus(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf; 
  BME82M131_SD_t  BME82M131_sd;
  BME82M131_status = BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf); 
  
  if(BME82M131_status)
  {
    _BME82M131_ConfReg.all = BME82M131_als_conf;
    BME82M131_sd = (BME82M131_SD_t)_BME82M131_ConfReg.ALS_SD;
   if(BME82M131_sd==BME82M131_POWER_ON)
   {
    return 0;
   }
   else if(BME82M131_sd==BME82M131_SHUT_DOWN)
   {
    return 1;
   }
  }
  else
  {
    return 0;
  } 
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's interrupt enable setting (ALS_INT_EN)
 * @param  sensorNmuber : Module Number
 * @retval isInterruptEnabled
            @arg  0-BME82M131_INT_DISABLE     
            @arg  1-BME82M131_INT_ENABLE 
 ************************************************************************************************************/
uint8_t BME82M131_isInterruptEnabled(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  BME82M131_INTEN_t BME82M131_ie;
  BME82M131_status = BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf); 
  if(BME82M131_status)
  {
    _BME82M131_ConfReg.all = BME82M131_als_conf;
    BME82M131_ie = (BME82M131_INTEN_t)_BME82M131_ConfReg.ALS_INT_EN;
    if(BME82M131_ie == BME82M131_INT_DISABLE)
    {
      return 0;
    }
    else if(BME82M131_ie == BME82M131_INT_ENABLE)
    {
      return 1;
    }
    else
    {
      return 0;
    }
  }
  else
  {
     return 0;
  } 
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's persistence protect number setting (ALS_PERS)
 * @param  sensorNmuber : Module Number
 * @retval Persistence
            @arg  0-BME82M131_PERS_1      
            @arg  1-BME82M131_PERS_2 
            @arg  2-BME82M131_PERS_4     
            @arg  3-BME82M131_PERS_8
 ************************************************************************************************************/
uint8_t BME82M131_getPersistence(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  BME82M131_PERS_t  BME82M131_pers;
  BME82M131_status = BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf); 
  if(BME82M131_status)
  {
    _BME82M131_ConfReg.all = BME82M131_als_conf;
    BME82M131_pers = (BME82M131_PERS_t)_BME82M131_ConfReg.ALS_PERS;
   if(BME82M131_pers==BME82M131_PERS_1)
   {
    return 0;
   }
   else if(BME82M131_pers==BME82M131_PERS_2)
   {
    return 1;
   }
   else if(BME82M131_pers==BME82M131_PERS_4)
   {
    return 2;
   }
   else if(BME82M131_pers==BME82M131_PERS_8)
   {
    return 3;
   }
  }
  else
  {
    return 0;
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's integration time setting (ALS_IT)
 * @param  sensorNmuber : Module Number
 * @retval IntegrationTime
            @arg  0-BME82M131_IT_25ms        
            @arg  1-BME82M131_IT_50ms
            @arg  2-BME82M131_IT_100ms    
            @arg  3-BME82M131_IT_200ms
            @arg  4-BME82M131_IT_400ms    
            @arg  5-BME82M131_IT_800ms
 ************************************************************************************************************/
uint8_t BME82M131_getIntegrationTime(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  BME82M131_IT_TIME_t BME82M131_it;
  BME82M131_status = BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  if(BME82M131_status)
  {
    _BME82M131_ConfReg.all = BME82M131_als_conf;
    BME82M131_it = ITTimeFromCONf((BME82M131_CONF_IT_TIME_t)_BME82M131_ConfReg.ALS_IT);
   if(BME82M131_it==BME82M131_IT_25ms)
   {
    return 0;
   }
   else if(BME82M131_it==BME82M131_IT_50ms)
   {
    return 1;
   }
   else if(BME82M131_it==BME82M131_IT_100ms)
   {
    return 2;
   }
   else if(BME82M131_it==BME82M131_IT_200ms)
   {
    return 3;
   }
   else if(BME82M131_it==BME82M131_IT_400ms)
   {
    return 4;
   }
   else if(BME82M131_it==BME82M131_IT_800ms)
   {
    return 5;
   }
  }
  else
  {
    return 0;
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's gain.
 * @param  sensorNmuber : Module Number
 * @retval Gain
            @arg  0-BME82M131_GAIN_1       
            @arg  1-BME82M131_GAIN_2
            @arg  2-BME82M131_GAIN_1_8     
            @arg  3-BME82M131_GAIN_1_4
 ************************************************************************************************************/
uint8_t BME82M131_getGain(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  BME82M131_GAIN_t  BME82M131_gain;
  BME82M131_status = BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  if(BME82M131_status)
  {
    _BME82M131_ConfReg.all = BME82M131_als_conf;
     BME82M131_gain = (BME82M131_GAIN_t)_BME82M131_ConfReg.ALS_GAIN;
   if(BME82M131_gain==BME82M131_GAIN_1)
   {
    return 0;
   }
   else if(BME82M131_gain==BME82M131_GAIN_2)
   {
    return 1;
   }
   else if(BME82M131_gain==BME82M131_GAIN_1_8)
   {
    return 2;
   }
   else if(BME82M131_gain==BME82M131_GAIN_1_4)
   {
    return 3;
   }
  }
  else
  {
    return 0;
  }
  return 0;
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's ALS high threshold window setting.
 * @param  sensorNmuber : Module Number
 * @retval threshold: high threshold window
 ************************************************************************************************************/
uint16_t BME82M131_getHighThreshold(uint8_t sensorNmuber)
{
  uint16_t BME82M131_threshold=0; 
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_HIGH_THRESHOLD;
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,7))
  { 
    if( _BME82M131_rx_buf[3] == 0)
    {
      BME82M131_threshold = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      return BME82M131_threshold;
    }
    else
    {
      return 0;
    }
  }
  else
  {
    return 0;
  }
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's ALS low threshold .
 * @param  sensorNmuber : Module Number
 * @retval threshold: low threshold 
 ************************************************************************************************************/
uint16_t BME82M131_getLowThreshold(uint8_t sensorNmuber)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_threshold=0;
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_LOW_THRESHOLD;

  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,7))
  {
    BME82M131_status = _BME82M131_rx_buf[3];  
    if(BME82M131_status == 0)
    {
      BME82M131_threshold = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      return BME82M131_threshold;
    }
    else
    {
       return 0;
    }
  }
  else
  {
    return 0;
  }
}


/*********************************************************************************************************//**
 * @brief  set the VEML7700's Power saving mode.
 * @param  sensorNmuber : Module Number
           mode:
              @arg BME82M131_POWER_SAVING_MODE1              
              @arg BME82M131_POWER_SAVING_MODE2              
              @arg BME82M131_POWER_SAVING_MODE3              
              @arg BME82M131_POWER_SAVING_MODE4
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setPowerSavingMode(uint8_t sensorNmuber, BME82M131_PSM_MODE_t mode)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_psm;
  
  BME82M131_getPsmReg(sensorNmuber, &BME82M131_psm);
  _BME82M131_PsmReg.all = BME82M131_psm;
  _BME82M131_PsmReg.PSM_MODE = (uint16_t)mode;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_POWER_SAVING_MODE;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_PsmReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_PsmReg.all & 0xff00) >> 8);
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else 
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  set the VEML7700's power saving mode status.
 * @param  sensorNmuber : Module Number
           psmen : 
              @arg BME82M131_POWER_SAVING_MODE_DISABLE            
              @arg BME82M131_POWER_SAVING_MODE_ENABLE            
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setPowerSavingModeStatus(uint8_t sensorNmuber, BME82M131_PSMEN_t psmen)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_psm;
  
  BME82M131_getPsmReg(sensorNmuber, &BME82M131_psm);
  _BME82M131_PsmReg.all = BME82M131_psm;
  _BME82M131_PsmReg.PSM_EN = (uint16_t)psmen;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_POWER_SAVING_MODE;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_PsmReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_PsmReg.all & 0xff00) >> 8);
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  } 
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's power on setting (ALS_SD)
 * @param  sensorNmuber : Module Number           
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_ALSOn(uint8_t sensorNmuber)
{
  if(BME82M131_setStatus(sensorNmuber,BME82M131_POWER_ON))
  {
    return true;
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's power down setting (ALS_SD)
 * @param  sensorNmuber : Module Number           
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_ALSDown( uint8_t sensorNmuber)
{
  if(BME82M131_setStatus(sensorNmuber,BME82M131_SHUT_DOWN))
  {
    return true;
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's interrupt(ALS_INT_EN)
 * @param  sensorNmuber : Module Number
           ie :
               @arg BME82M131_INT_DISABLE
               @arg BME82M131_INT_ENABLE  
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setInterruptStatus(uint8_t sensorNmuber, BME82M131_INTEN_t ie)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  
  BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  _BME82M131_ConfReg.all = BME82M131_als_conf;
  _BME82M131_ConfReg.ALS_INT_EN = (uint16_t)ie;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);
  
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];  
    if(BME82M131_status == 0)
    {
     return true;  
    }
    else
    {
      return false;
    }
 }
 else
 {
  return false;
 }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's persistence protect number setting (ALS_PERS)
 * @param  sensorNmuber : Module Number
           pers :
              @arg BME82M131_PERS_1              
              @arg BME82M131_PERS_2                      
              @arg BME82M131_PERS_4              
              @arg BME82M131_PERS_8    
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setPersistence(uint8_t sensorNmuber, BME82M131_PERS_t pers)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  
  BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  _BME82M131_ConfReg.all = BME82M131_als_conf;
  _BME82M131_ConfReg.ALS_PERS = (uint16_t)pers;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);
  
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
     BME82M131_status = _BME82M131_rx_buf[3];
     if(BME82M131_status==0)  
     {
        return true;
     }
     else
     {
        return false;
     }
  }
  else
  {
  return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's integration time setting (ALS_IT)
 * @param  sensorNmuber : Module Number
           it :
                @arg BME82M131_IT_25ms               
                @arg BME82M131_IT_50ms           
                @arg BME82M131_IT_100ms      
                @arg BME82M131_IT_200ms      
                @arg BME82M131_IT_400ms      
                @arg BME82M131_IT_800ms     
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setIntegrationTime(uint8_t sensorNmuber, BME82M131_IT_TIME_t it)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  
  BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  _BME82M131_ConfReg.all = BME82M131_als_conf;
  _BME82M131_ConfReg.ALS_IT = (uint16_t)configITTime(it);
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);
  
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's gain.
 * @param  sensorNmuber : Module Number
           gain : 
                @arg  BME82M131_GAIN_1              
                @arg  BME82M131_GAIN_2         
                @arg  BME82M131_GAIN_1_8           
                @arg  BME82M131_GAIN_1_4    
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setGain(uint8_t sensorNmuber, BME82M131_GAIN_t gain)
{
  uint8_t BME82M131_status;
  uint16_t BME82M131_als_conf;
  
  BME82M131_getAlsConfig(sensorNmuber, &BME82M131_als_conf);  
  _BME82M131_ConfReg.all = BME82M131_als_conf;
  _BME82M131_ConfReg.ALS_GAIN = (uint16_t)gain;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);
  
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
 else
 {
  return false;
 }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's ALS high threshold window setting.
 * @param  sensorNmuber : Module Number
           threshold : 0x0000~0xffff.   
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setHighThreshold(uint8_t sensorNmuber, uint16_t threshold)
{
  uint8_t BME82M131_status=0; 
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_HIGH_THRESHOLD;
  _BME82M131_tx_buf[4] = (uint8_t)(threshold & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((threshold & 0xff00) >> 8);
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's ALS low threshold window setting.
 * @param  sensorNmuber : Module Number
           threshold : 0x0000~0xffff.    
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setLowThreshold(uint8_t sensorNmuber, uint16_t threshold)
{
  uint8_t BME82M131_status;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_LOW_THRESHOLD;
  _BME82M131_tx_buf[4] = (uint8_t)(threshold & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((threshold & 0xff00) >> 8);
  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
 else
 {
  return false;
 } 
}


		
/* Private functions ---------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief  Run through initialization procedure for each sensor connected on the bus
 * @param  void   
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_initialize()
{
  uint8_t BME82M131_status=0;  
	uint8_t BME82M131_i = 0;
  _BME82M131_ConfReg.all = 0x0000; // Clear the reserved bits
  _BME82M131_ConfReg.ALS_SD = BME82M131_POWER_ON;
  _BME82M131_ConfReg.ALS_INT_EN = BME82M131_INT_DISABLE;
  _BME82M131_ConfReg.ALS_PERS = BME82M131_PERS_1;
  _BME82M131_ConfReg.ALS_IT = configITTime(BME82M131_IT_100ms);
  _BME82M131_ConfReg.ALS_GAIN = BME82M131_GAIN_1;  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);  
  for(BME82M131_i = 1; BME82M131_i <= _BME82M131_SumOfModules; BME82M131_i++)
  {
    _BME82M131_delay(20);
    BME82M131_sendData(BME82M131_i, 7, _BME82M131_tx_buf);
    _BME82M131_delay(20);
    if(BME82M131_readBytes(_BME82M131_rx_buf,5))
    {
      BME82M131_status=_BME82M131_rx_buf[3];
      if(BME82M131_status==0)
      {
        return true;
      }
      else 
      {
        return false;
      }
     }
     else
     {
      return false;
     }
  }
  return false;
}

/*********************************************************************************************************//**
 * @brief  Check that the sensor is responding on the I2C bus
 * @param  void   
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_isConnected()
{
  uint8_t BME82M131_triesBeforeGiveup = 5;
	uint8_t BME82M131_x = 0;
	
  for (BME82M131_x = 0; BME82M131_x < BME82M131_triesBeforeGiveup; BME82M131_x++)
  {
	  if(BME82M131_getNumber() != 0)
		{
			return true;
		}
		_BME82M131_delay(100);
	}
	return false;

}

/*********************************************************************************************************//**
 * @brief  Set the VEML7700's shut down setting (ALS_SD)
 * @param  sensornumber : Module Number
           sd : 
                @arg BME82M131_POWER_ON
                @arg BME82M131_SHUT_DOWN
 * @retval result
              @arg ture 
              @arg false
 ************************************************************************************************************/
bool BME82M131_setStatus(uint8_t sensornumber, BME82M131_SD_t sd)
{
  uint16_t BME82M131_als_conf;
  
  BME82M131_getAlsConfig(sensornumber, &BME82M131_als_conf);  
  _BME82M131_ConfReg.all = BME82M131_als_conf;
  _BME82M131_ConfReg.ALS_SD = (uint16_t)sd;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_W_ALS_CONFIG;
  _BME82M131_tx_buf[4] = (uint8_t)(_BME82M131_ConfReg.all & 0x00ff);
  _BME82M131_tx_buf[5] = (uint8_t)((_BME82M131_ConfReg.all & 0xff00) >> 8);
  
  _BME82M131_delay(20);
  BME82M131_sendData(sensornumber, 7, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,5))
  { 
  return true;
  }
 else
 { 
  return false;
 }
}

/*********************************************************************************************************//**
 * @brief  Write data to the module through I2C
 * @param  id: The module number
           len:Length of data to be written
			     *par: Write to an array of data
 * @retval void
 ************************************************************************************************************/
void BME82M131_sendData(uint8_t id, uint8_t len, uint8_t par[])
{
	uint8_t BME82M131_i = 0;
 /* Array frame format
    MID | ID | LEN | CMD/STATUS | DATA0~n | CHECHSUM*/
	par[0] = BME82M131_MODULE_MID;
	par[1] = id;
	par[2] = len - 3;
	par[len - 1] = 0;
	
	for(BME82M131_i = 0; BME82M131_i < len - 1; BME82M131_i++)
	{
		par[len - 1] += par[BME82M131_i];
	}
  BME82M131_writeBytes(par,len);
}

/*********************************************************************************************************//**
 * @brief  writeBytes.
 * @param  wbuf[] : Data to be written.
           wlen : Data length.
 * @retval void
 ************************************************************************************************************/
void BME82M131_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
    I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
    I2CMaster_Typedef gI2CMaster = {0x00};

    gI2CMaster.uSlaveAddress = BME82M131_MASTERADDR;
    gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
    gI2CMaster.Tx.uLength = wlen;
    gI2CMaster.uTimeout_us = 30000;
    gI2CMaster.isDynamicOnOff_I2C = FALSE;

    I2CResult = I2C_MASTER_GOING;
    I2CMaster_Write(gBME82M131_WIRE,&gI2CMaster);
    do {
      I2CResult = I2CMaster_GetStatus(gBME82M131_WIRE);
    } while (I2CResult == I2C_MASTER_GOING);
}

/*********************************************************************************************************//**
 * @brief  Read the data of the module through I2C
 * @param  rlen: The length of the data read
    			 rbuf[]: Store the read data
 * @retval result
             @arg true 
             @arg flase
 ************************************************************************************************************/
bool BME82M131_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	uint8_t BME82M131_i = 0, BME82M131_checkSum = 0;
	{
     I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
     I2CMaster_Typedef I2CMaster = { 0 };

     I2CMaster.uSlaveAddress = BME82M131_MASTERADDR;
     I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
     I2CMaster.Rx.uLength = rlen;
     I2CMaster.uTimeout_us = 30000;
     I2CMaster.isDynamicOnOff_I2C = FALSE;

     I2CMaster_Read(gBME82M131_WIRE, &I2CMaster);
	
     do {
       I2CResult = I2CMaster_GetStatus(gBME82M131_WIRE);
     } while (I2CResult == I2C_MASTER_GOING);
	 }
  /* Check Sum */
  for (BME82M131_i = 0; BME82M131_i < (rlen - 1); BME82M131_i++)
  {
    BME82M131_checkSum += rbuf[BME82M131_i];
  }
  if (BME82M131_checkSum == rbuf[rlen - 1])
  {
    return true; // Check correct
  }
  else
  {
    return false; // Check error
  }
}

/*********************************************************************************************************//**
 * @brief  Get register data
 * @param  sensornumber:Module Number
    			 *als_conf: Stores the value. 
 * @retval result
             @arg true 
             @arg flase
 ************************************************************************************************************/
bool BME82M131_getAlsConfig(uint8_t sensornumber,uint16_t *als_conf)
{
  uint8_t BME82M131_status;
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_ALS_CONFIG;

  _BME82M131_delay(20);
  BME82M131_sendData(sensornumber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,7))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status==0)
    {
      *als_conf = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      return  true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's  value of the PSM register.
 * @param  sensornumber:Module Number
    			 *psm : Stores the value.   
 * @retval result
             @arg true 
             @arg flase
 ************************************************************************************************************/
bool BME82M131_getPsmReg(uint8_t sensorNmuber, uint16_t *psm)
{
  uint8_t BME82M131_status;
  
  _BME82M131_tx_buf[3] = _BME82M131_CMD_R_POWER_SAVING_MODE;

  _BME82M131_delay(20);
  BME82M131_sendData(sensorNmuber, 5, _BME82M131_tx_buf);
  _BME82M131_delay(20);
  if(BME82M131_readBytes(_BME82M131_rx_buf,7))
  {
    BME82M131_status = _BME82M131_rx_buf[3];
    if(BME82M131_status == 0)
    {
      *psm = (uint16_t)_BME82M131_rx_buf[4] + ((uint16_t)_BME82M131_rx_buf[5] << 8);
      return true;
    }
    else
    {
      psm = 0;
      return false;
    }
  }
  else
  {
    psm = 0;
    return false;
  }  
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's gain as printable text
 * @param  it:CONF_IT_25ms = 0b1100,      // ALS intetgration time 25ms
              @arg BME82M131_CONF_IT_50ms = 0b1000,      // ALS intetgration time 50ms
              @arg BME82M131_CONF_IT_100ms = 0b0000,     // ALS intetgration time 100ms
              @arg BME82M131_CONF_IT_200ms = 0b0001,     // ALS intetgration time 200ms
              @arg BME82M131_CONF_IT_400ms = 0b0010,     // ALS intetgration time 400ms
              @arg BME82M131_CONF_IT_800ms = 0b0011,     // ALS intetgration time 800ms  
 * @retval result
             @arg true 
             @arg flase
 ************************************************************************************************************/
BME82M131_IT_TIME_t ITTimeFromCONf(BME82M131_CONF_IT_TIME_t it)
{
  switch(it)
  {
    case BME82M131_CONF_IT_100ms:
    {
      return BME82M131_IT_100ms;
    }
    case BME82M131_CONF_IT_200ms:
    {
      return BME82M131_IT_200ms;
    }
    case BME82M131_CONF_IT_400ms:
    {
      return BME82M131_IT_400ms;
    }
    case BME82M131_CONF_IT_800ms:
    {
      return BME82M131_IT_800ms;
    }
    case BME82M131_CONF_IT_50ms:
    {
      return BME82M131_IT_50ms;
    }
    case BME82M131_CONF_IT_25ms:
    {
      return BME82M131_IT_25ms;
    }
   default:
   {
      return BME82M131_IT_INVALID;
   }
  }
}

/*********************************************************************************************************//**
 * @brief  Get the VEML7700's gain as printable text
 * @param  it:CONF_IT_25ms = 0b1100,      // ALS intetgration time 25ms
              @arg BME82M131_CONF_IT_50ms = 0b1000,      // ALS intetgration time 50ms
              @arg BME82M131_CONF_IT_100ms = 0b0000,     // ALS intetgration time 100ms
              @arg BME82M131_CONF_IT_200ms = 0b0001,     // ALS intetgration time 200ms
              @arg BME82M131_CONF_IT_400ms = 0b0010,     // ALS intetgration time 400ms
              @arg BME82M131_CONF_IT_800ms = 0b0011,     // ALS intetgration time 800ms 
 * @retval result
              @arg 0x00: BME82M131_CMD_TRANSFER_SUCCESS
              @arg 0x40: BME82M131_CHECKSUM_ERROR
              @arg 0x80: BME82M131_INSTRUCTION_NOT_SUPPOR
              @arg 0xA0: BME82M131_SLAVE_NO_RESPONSE
 ************************************************************************************************************/
BME82M131_CONF_IT_TIME_t configITTime(BME82M131_IT_TIME_t it)
{
  switch(it)
  {
    case BME82M131_IT_25ms:
    {
      return BME82M131_CONF_IT_25ms;
    }
    case BME82M131_IT_50ms:
    {
      return BME82M131_CONF_IT_50ms;
    }
    case BME82M131_IT_100ms:
    {
      return BME82M131_CONF_IT_100ms;
    }
    case BME82M131_IT_200ms:
    {
      return BME82M131_CONF_IT_200ms;
    }
    case BME82M131_IT_400ms:
    {
      return BME82M131_CONF_IT_400ms;
    }
    case BME82M131_IT_800ms:
    {
      return BME82M131_CONF_IT_800ms;
    }
   default:
   {
      return BME82M131_CONF_IT_INVALID;
   }
  } 
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BME82M131_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}

