/*************************************************
File:               BM22O2421_A.cpp
Description:        Describe the function of BestModulesCorp Servo module BM22O2421-A application library
History:            V1.00   -- initial version；2021/08/22；Arduino IDE :  ≥v1.8.13
Author:             Best Modules Corp
**************************************************/

#include "BM22O2421_A.h"

BM22O2421_A::BM22O2421_A(uint8_t TxRxPin)   // Creates a new object to interface with the Servo
  : _Serial9N1(TxRxPin)
{
  _Serial9N1.begin(115200);                 // The BM22O2421 support only  baud rate 115200, 
};

BM22O2421_A::~BM22O2421_A()                 // destructor
{
  _Serial9N1.~SoftwareSerial_9N1_OneWire();
}

/*************************************************
Description: Get the servo response and return the servo status

Return: 
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail
*************************************************/
uint8_t BM22O2421_A::Response()
{ 
  uint8_t Return_data[10],Return_data_counter,data,e;
  delay(10);
  Return_data_counter=0;                                
  while(_Serial9N1.available())       // Check the connection for receivables
  {
    _Serial9N1.readbit9(&data);       // Read the data sent from the servo
    Return_data[Return_data_counter++]=data;  
  }
    _Serial9N1.OneWireRxModEnd();   
    
  switch(Return_data[2])							// Return_data[2] is the status
    {
      case 0x00: e = ACK_SUCCESSFULLY; break;
      case 0x02: e = NACK_LOW_VOLTAGE; break;
      case 0x04: e = NACK_STALL_PROTECTION; break;
      case 0x08: e = NACK_ANGLE_ERROR; break;
      case 0x40: e = NACK_CHECKSUM_ERROR; break;
      case 0x80: e = NACK_UNSUPPORTED_COMMAND; break;    
      default:   e = FAIL; break;				
    }
    return e;
}

/*************************************************
Description: GetCommandPacket6 represents the 8 byte command that we send to the servo

Parameter:          
  Eid: eid of servo (1~15)
  Command: the command to be executed
  Mode: Rotate or Angle
  Time: time to complete the action

Return:             
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail                 
*************************************************/
uint8_t BM22O2421_A::GetCommandPacket6(uint8_t Eid,uint8_t Command,int16_t Mode,uint16_t Time)
{ 
  uint8_t TimeH=(uint8_t)(Time>>8);       // High Byte of Time
  uint8_t TimeL=(uint8_t)(Time&0xFF);     // Low Byte of Time
  uint8_t ModeH=(uint8_t)(Mode>>8);       // High Byte of Rotate/Angle
  uint8_t ModeL=(uint8_t)(Mode&0xFF);     // Low Byte of Rotate/Angle
  uint8_t CheckSum=~((MID)+(Eid+TLEN6)+(Command)+(ModeL)+(ModeH)+(TimeL)+(TimeH));
  _Serial9N1.OneWireTxMod(115200);
  _Serial9N1.writebit9(MID,1);        // MID
  _Serial9N1.writebit9(Eid+TLEN6,0);  // 0x60 + Eid of the servo, TLEN6 = 0x60
  _Serial9N1.writebit9(Command,0);    // Command: ROTATE, ANGLE
  _Serial9N1.writebit9(ModeL,0);      // direction/speed for ROTATE, angle for ANGLE                  
  _Serial9N1.writebit9(ModeH,0);                     
  _Serial9N1.writebit9(TimeL,0);      // time to complete the action
  _Serial9N1.writebit9(TimeH,0); 
  _Serial9N1.writebit9(CheckSum,0);   // checksum               
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200); 
  return Response(); 
}

/*************************************************
Description: Reset the servo

Parameter:          
  Eid: eid of servo (1~15)

Return:             
  0x00 : successful
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail     
*************************************************/
uint8_t BM22O2421_A::Reset(uint8_t Eid)  //If Eid = 1, 4 bytes (10h,21h,00h,CEh)are transmitted.
{  
  uint8_t CheckSum=~((MID)+(Eid+0x20)+(RESET));
  _Serial9N1.OneWireTxMod(115200);
  _Serial9N1.writebit9(MID,1);           // MID, 0x10
  _Serial9N1.writebit9(Eid+0x20,0);      // 0x20 + Eid of the servo
  _Serial9N1.writebit9(RESET,0);         // instr, 0 for Reset 
  _Serial9N1.writebit9(CheckSum,0);      // checksum   
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200); 
  return Response(); 
}

/*************************************************
Description: Let the servo enter standby

Parameter:
  Eid: Eid of dedicae servo (1~15)to enter standby status
  if you want to let all servos standby, set Eid = 0

Return:             
  The servo does not respond this command. 
*************************************************/
void BM22O2421_A::Standby()   //Broadcast command, 4 bytes (00h,20h,01h,DEh)are transmitted. 
{
  uint8_t CheckSum=~((0)+(0x20)+(STANDBY));
  _Serial9N1.OneWireTxMod(115200);
  _Serial9N1.writebit9(0,1);          // MID should be 0 for a broadcast command
  _Serial9N1.writebit9(0x20,0);       // Eid should be 0 for a broadcast command
  _Serial9N1.writebit9(0x01,0);       // instr, 0x01 for Standby
  _Serial9N1.writebit9(CheckSum,0);   // checksum   
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200); 
}

/*************************************************
Description: 
  Synchronous execution of the previous command(rotate, angle) for all servos.
  This is a broadcast command

Return:
  The servo does not respond this command.
*************************************************/
void BM22O2421_A::SyncAction()   //Broadcast command, 4 bytes (00h,20h,02h,DDh)are transmitted. 
{ 
  uint8_t  CheckSum=~((0)+(0x20)+(SYNCACTION));
  _Serial9N1.OneWireTxMod(115200);
  _Serial9N1.writebit9(0,1);          // MID should be 0 for a broadcast command
  _Serial9N1.writebit9(0x20,0);       // Eid should be 0 for a broadcast command
  _Serial9N1.writebit9(0x02,0);       // instr, 0x02 for SyncAction
  _Serial9N1.writebit9(CheckSum,0);   // the checksum is 0xdd
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200); 
}

/*************************************************
Description: Get the firmware version 

Parameter:          
  Eid: Eid of servo (1~15)

Return: a 32-bit value         
  bit 31-24: 0
  bit 23-16: servo status
  bit 15-8: Firmware version low byte
  bit 7-0: Firmware version high byte
*************************************************/
uint32_t BM22O2421_A::GetFWVer(uint8_t Eid)  //If Eid = 1, 6 bytes (10h,21h,03h,CBh)are transmitted.  
{ 
  uint8_t CheckSum=~((MID)+(Eid+0x20)+(GETVERSION));
  uint8_t Return_data[10],Return_data_counter=0,data;
  uint32_t rtn=0;
  _Serial9N1.OneWireTxMod(115200);
  _Serial9N1.writebit9(MID,1);                 // MID, 0x10
  _Serial9N1.writebit9(Eid+0x20,0);            // 0x20 + Eid of the servo
  _Serial9N1.writebit9(GETVERSION,0);          // instr, 0x03 for GetFWVer
  _Serial9N1.writebit9(CheckSum,0);            
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200); 

  delay(10);
  while(_Serial9N1.available())       // Check the connection for receivables
  {
    _Serial9N1.readbit9(&data);       // Read the data sent from the servo
    Return_data[Return_data_counter++]=data;  
  }
    _Serial9N1.OneWireRxModEnd();   
  rtn+=(uint32_t)Return_data[2];
  rtn=rtn<<8;
  rtn+=(uint32_t)Return_data[3];
  rtn=rtn<<8;
  rtn+=(uint32_t)Return_data[4];
  return rtn;
}

/*************************************************
Description: Set Eid of the servo

Parameter:          
  Eid: Eid of servo (1~15)

Return:             
  0x00 : successful
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail     
*************************************************/
uint8_t BM22O2421_A::SetEid(uint8_t Eid)   //Set Eid = 1, 4 bytes (00h,20h,81h,5Eh)are transmitted.
{ 
  uint8_t CheckSum=~((0)+(SETEID+Eid)+(0x20));              
  _Serial9N1.OneWireTxMod(115200);      
  _Serial9N1.writebit9(0,1);            // MID should be 0 for a broadcast command
  _Serial9N1.writebit9(0x20,0);         // Eid should be 0 for a broadcast command
  _Serial9N1.writebit9(SETEID+Eid,0);   // instr, =SetEid (0x80) + Eid (the Eid to be set)
  _Serial9N1.writebit9(CheckSum,0);     // checksum   
  _Serial9N1.OneWireTxModEnd();  
  _Serial9N1.OneWireRxMod(115200);
  return Response();  
}

/*************************************************
Description: set the servo rotate direction, speed and duration

Parameter:          
  Eid: Eid of servo (1~15)
  Direction: the direction and speed of rotation (-1000~1000)
    -1000 ~ -1: counterclockwise, with speed 1 - 1000 (1000 is the fastest)
    1 ~ 1000: cloclwise, with speed 1 - 1000
  Time: the duration of rotation (0~65535)ms 

Return:             
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail                 
*************************************************/
uint8_t BM22O2421_A::Rotate(uint8_t Eid,int16_t Speed,uint16_t Time)  //If Eid = 1 / Speed = 1000 / Time = 1000, 4 bytes (10h,61h,08h,E8h,03h,E8h,03h,B0h)are transmitted.
{
  return GetCommandPacket6(Eid,ROTATE,Speed,Time); 
}

/*************************************************
Description: set the servo to rotate to an angle and time of reaching the angle

Parameter:          
  Eid: Eid of servo (1~15)
  Angle: the angle of rotation (0~240)
  Time: the time of reaching the angle(0~65535)ms  

Return:             
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail
*************************************************/
uint8_t BM22O2421_A::Angle(uint8_t Eid,int16_t Angle,uint16_t Time)  //If Eid = 1 / Angle = 0 / Time = 1000, 4 bytes (10h,61h,09h,00h,00h,E8h,03h,9Ah)are transmitted.
{
  return GetCommandPacket6(Eid,ANGLE,Angle,Time);
}

/*************************************************
Description:
  set the servo rotate direction, speed and duration(must be used with SyncAction function)
  The servo will not action until it receives the SyncAction command

Parameter:          
  Eid: Eid of servo (1~15)
  Direction: the direction and speed of rotation (-1000~1000)
    -1000 ~ -1: counterclockwise, with speed 1 - 1000 (1000 is the fastest)
    1 ~ 1000: cloclwise, with speed 1 - 1000
  Time: the duration of rotation (0~65535)ms 
  
Return:             
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail
*************************************************/
uint8_t BM22O2421_A::DelayRotate(uint8_t Eid,int16_t Speed,uint16_t Time)  //If Eid = 1 / Speed = 1000 / Time = 1000, 4 bytes (10h,61h,28h,E8h,03h,E8h,03h,90h)are transmitted.
{
  return GetCommandPacket6(Eid,DELAYROTATE,Speed,Time); 
}

/*************************************************
Description: 
  set the servo to rotate to an angle and time of reaching the angle (must be used with SyncAction function)
  The servo will not action until it receives the SyncAction command

Parameter:          
  Eid: Eid of servo (1~15)
  Angle: the angle of rotation (0~240)
  Time: the time of reaching the angle,(0~65535)ms  

Return:             
  0x00 : successful
  0x02 : system voltage is lower than 6.5V
  0x04 : stall protection
  0x08 : angle is incorrect
  0x40 : checkSum error
  0x80 : unsupported command   
  0xAA : fail
*************************************************/
uint8_t BM22O2421_A::DelayAngle(uint8_t Eid,int16_t Angle,uint16_t Time)  //If Eid = 1 / Angle = 0 / Time = 1000, 4 bytes (10h,61h,29h,00h,00h,E8h,03h,7Ah)are transmitted.
{
  return GetCommandPacket6(Eid,DELAYANGLE,Angle,Time);
}
