/*****************************************************************
  File:             BM22S4221-1.cpp
  Author:           BEST MODULES CORP.
  Description:      Communication and operation function with module
  History：
  V1.0.1-- initial version；2022-09-06；Arduino IDE : v1.8.13
******************************************************************/
#include  "BM22S4221-1.h"
uint8_t receiveBuffer[25];

/**********************************************************
Description: Select the hardware serial port you need to use
Parameters:  *theSerial：hardware serial 
             BMduino optional:serial(default) serial1/seria2/seria3/seria4
             UNO optional:serial(default)  
**********************************************************/
BM22S4221_1::BM22S4221_1(uint8_t statusPin,HardwareSerial*theSerial)
{
  _serial = theSerial;
  _softSerial = NULL;
  _statusPin = statusPin;
}
/**********************************************************
Description: Select the software serial port RX TX you need to use
Parameters:  rxPin:RX pin on the development board
             txPin:TX pin on the development board
**********************************************************/
BM22S4221_1::BM22S4221_1(uint8_t statusPin,uint8_t rxPin, uint8_t txPin)
{
  _serial = NULL;
  _statusPin = statusPin;
  _rxPin = rxPin;
  _txPin = txPin;
  _softSerial = new SoftwareSerial(_rxPin, _txPin);
}
/**********************************************************
Description: Set serial baud rate
Parameters:  uartBaud：9600(default)
Return:      none
**********************************************************/
void BM22S4221_1::begin()
{
  if (_softSerial != NULL) {_softSerial->begin(UART_BAUD);}
  else  {_serial->begin(UART_BAUD);}
}

/**********************************************************
U00
Description: reset the MCU in the module
Parameters:  none
Return:      0: success
             1: failed   
**********************************************************/
uint8_t BM22S4221_1::resetModule()
{ 
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xAF, 0x00, 0x00, 0x51};
  wirteData(sendCmd, 4);
  delay(30);// wait module response
  readData(receiveBuffer);
  delay(100);// wait module to execute initialization.
  if (debug){
    Serial.println("U00: reset module");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;  
}
/**********************************************************
U01
Description: Query the FW version and production date.
             The FW version number and production date are both 8421 BCD code.
Parameters:  buff(12 bytes): Store the FW version and production date
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::getVerDate(uint8_t buff[])
{
  uint8_t sum = 0; 
  uint8_t sendCmd[4] = {0xAD, 0x00, 0x00, 0x53};
  wirteData(sendCmd, 4);
  delay(30); //wait module response
  readData(buff);
  if (debug){
    Serial.println("U01: get version and date");
    for (uint8_t i=0; i<12; i++)  {Serial.println(buff[i], HEX);}
  }  
  for (uint8_t i=0; i<11; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[11] == (uint8_t)(~sum+1)) return 0;
  else  return   1; 
}
/**********************************************************
U02
Description: Get all current data of the module
Parameters:  buff(25byte): for storing the data
Return: 0: success
        1: failed
**********************************************************/
uint8_t BM22S4221_1::getInfoPackage(uint8_t buff[])
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xAC, 0x00, 0x00, 0x54};
  wirteData(sendCmd, 4);
  delay(70);//wait module response
  readData(buff);
  if (debug){
    Serial.println("U02: get module information");
    for (uint8_t i=0; i<25; i++)  {
      Serial.print(i); 
      Serial.print(" : 0x");  
      Serial.println(buff[i], HEX);
    }
  }  
  for (uint8_t i=0; i<24; i++) {sum += buff[i];}  
  if (buff[24] == (uint8_t)(~sum+1)) return 0;
  else return  1;
}
/**********************************************************
U03
Description: restore the module to factory default settings
Parameters:  none
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::restoreDefault()
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xA0, 0x00, 0x00, 0x60};
  wirteData(sendCmd, 4);
  delay(120);//wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("U03: restore factory default settings");
    for (uint8_t i=0; i<8; i++)  {Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;
}
/**********************************************************
R00
Description: Query whether the UART auto-Tx is enabled
Parameters:  none
Return:      0: auto-Tx disabled
             8: auto-Tx enabled
             1: failed
**********************************************************/
uint8_t BM22S4221_1::isAutoTx()
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xD0, 0x1B, 0, 0x15}; 
  wirteData(sendCmd, 4);
  delay(30);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R00: get auto-tx status");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return receiveBuffer[6];
  else return  1;
}
/**********************************************************
R01
Description: Get the alarm output level of the STATUS pin
Parameters: None
Return: 8 : STATUS pin is high when alarming, low when not alarming
        0 : STATUS pin is low when alarming, high when not alarming
        1 : failed
**********************************************************/
uint8_t BM22S4221_1::getStatusPinActiveMode()
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xD0, 0x1C, 0x00, 0x14};  
  wirteData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R01: get alarm level");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return receiveBuffer[6];
  else return  1;  
}
/**********************************************************
R02
Description: Query the A/D value of the internal VBG voltage 
             VBG A/D value = 1.25V/VDD×256
Parameters: None
Return:    the A/D value of VBG
           0: failed
**********************************************************/
uint8_t BM22S4221_1::getVBG()
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xD2, 0x4C, 0x00, 0xE2};  
  wirteData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R02: get VBG");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return receiveBuffer[6];
  else return  0;    
}
/**********************************************************
W00
Description: Enable/Disable sending out device information automatically via UART
Parameters: state=08, enable 
            state=0, disable (default)
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setAutoTx(uint8_t state)
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xE0, 0x1B, 0, 0};  
  sendCmd[2] = state;
  sum = 0xE0 + 0x1B + state;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W00: Enable/Disable AutoTx");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;   
}
/**********************************************************
W01
Description: Modify device alarm output level
Parameters: state=08, status pin is high when alarming, low when not alarming (default)
            state=0, status pin is low when alarming, high when not alarming
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setStatusPinActiveMode(uint8_t state)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x1C, 0, 0};  
  sendCmd[2] = state;
  sum = 0xE0 + 0x1C + state;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W01: set staus pin level");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;           
}
/**********************************************************
W02
Description: Modify Internal OPA Gain
Parameters:  value,range is 0 to 31, default 31
             OPA gain=128 + value*8
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setOpaGain(uint8_t value)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x05, 0, 0};  
  sendCmd[2] = value;
  sum = 0xE0 + 0x05 + value;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W02: set OPA gain");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;        
}
/**********************************************************
W03
Description: Modify the detection deviation value
             deviation value:change amount of alarm detection
Parameters:  Threshold, range 15~120
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setAlarmThreshold(uint8_t Threshold)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x07, 0, 0};  
  sendCmd[2] = Threshold;
  sum = 0xE0 + 0x07 + Threshold;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W03: set alarm threshold");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;    
}
/**********************************************************
W04
Description: set alarm detection delay time          
Parameters:  time, the actual delay time is (time/2) seconds. default 6 (i.e. 3 seconds)
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setAlarmDetectDelay(uint8_t time)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x08, 0, 0};  
  sendCmd[2] = time;
  sum = 0xE0 + 0x08 + time;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W04: set alarm detect delay time");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;    
}
/**********************************************************
W05
Description: Set the output duration of alarm signal (status pin)           
Parameters:  time, the actual output time is (time/2) seconds. default 6 (i.e. 3 seconds)
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setAlarmOutputTime(uint8_t time)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x09, 0, 0};  
  sendCmd[2] = time;
  sum = 0xE0 + 0x09 + time;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W05: set alarm output duration");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;  
}
/**********************************************************
W06
Description: Modify preheating time 
Parameters:  time, value = 60 ~ 255, the actual preheat time is (time/2) seconds.
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::setPreheaTime(uint8_t time)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x0C, 0, 0};  
  sendCmd[2] = time;
  sum = 0xE0 + 0x0C + time;
  sendCmd[3] = ~sum + 1;
  wirteData(sendCmd, 4);
  delay(100);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W06: set preheat time");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;
}
/**********************************************************
Description: Read the data automatically output by the module
Parameters:  buff: buffer for storing the received data 
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S4221_1::autoRx(uint8_t buff[])
{
  uint8_t sum=0,num=0;
  clear_UART_FIFO();
  while (num < 25){
    if (_softSerial != NULL) {num = _softSerial->available();}
    else if (_serial != NULL){num = _serial->available();}
  }
  for (uint8_t i=0;i<25;i++){
    if (_softSerial != NULL) {buff[i] = _softSerial->read();}
    else if (_serial != NULL){buff[i] = _serial->read();}
  }
  for(uint8_t i=0;i<24;i++){sum += buff[i];}
  sum = ~sum + 1;
  if (debug){
    Serial.print("Auto Rx  ");
    Serial.println(num);
    for(uint8_t i=0;i<24;i++){
      Serial.print(i); Serial.print(": 0x"); Serial.println(buff[i], HEX);
    }
    Serial.println(sum, HEX);
    Serial.println("");  
  }
  if ((sum == buff[24]) && (buff[0]==0xAA)){return 0;}
  else return 1;
}
/**********************************************************
Description: UART wirteData
             Automatically determine whether a hardware or software serial  is used
Parameters:  buff: array where the data is stored to send
             length: the number of bytes to be sent from the array.
Return:      none
**********************************************************/
void  BM22S4221_1::wirteData(uint8_t buff[], uint8_t length)
{
  clear_UART_FIFO();  
  if (_softSerial != NULL) {_softSerial->write(buff, length);}  
  else {_serial->write(buff, length);}
}
/**********************************************************
Description: Clear UART FIFO
Parameters: None
Return: None
**********************************************************/
void BM22S4221_1::clear_UART_FIFO()
{
  if (_softSerial != NULL)
  {
    while (_softSerial->available() > 0) {_softSerial->read();}
  }
  if (_serial != NULL)
  {
    while (_serial->available() > 0) {_serial->read();}
  }
}
/**********************************************************
Description: UART readData
Parameters:       buff:Variables for storing Data to be read
             datalength:Length of data plus command
Return:      none
Others:
**********************************************************/
void  BM22S4221_1::readData(uint8_t buff[])
{
  uint8_t i=0;
  if (_softSerial != NULL)
  {
    while(_softSerial->available()>0)
    {
      buff[i] = _softSerial->read();
      i++;
    }
  }
  else
  {
    while(_serial->available()>0)
    {
      buff[i] = _serial->read();
      i++;
    }
  }
}
